<?php

namespace App\Http\Controllers\Writer;

use App\Models\User;
use Inertia\Inertia;
use App\Models\Subject;
use App\Models\Language;
use Illuminate\Http\Request;
use App\Models\EducationLevel;
use App\Models\Specialization;
use App\Models\WriterCitationStyle;
use App\Models\WriterLevel;
use App\Models\WriterPaymentMethod; // Add this
use App\Http\Controllers\Controller;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Storage;
use App\Notifications\ProfileUpdated;

class WriterProfileController extends Controller
{
    /**
     * Show the form for editing the writer profile.
     */
    // public function edit()
    // {
    //     $user = Auth::user();
    //     $writerProfile = $user->writerProfile()->with([
    //         'subjects',
    //         'languages',
    //         'citationStyles',
    //         'paymentAccounts.paymentMethod' // Load payment accounts
    //     ])->first();

    //     $isFullyQualified = $writerProfile->isApproved();

    //     // Prepare data for form
    //     $subjects = Subject::select('id', 'name')->orderBy('name')->get();
    //     $languages = Language::select('id', 'name')->orderBy('name')->get();
    //     $citationStyles = WriterCitationStyle::select('id', 'name')->orderBy('name')->get();
    //     $educationLevels = EducationLevel::select('id', 'name')->orderBy('name')->get();
    //     $specializations = Specialization::select('id', 'name')->orderBy('name')->get();
    //     $writerLevels = WriterLevel::select('id', 'name')->orderBy('name')->get();
    //     $paymentMethods = WriterPaymentMethod::where('is_active', true)->get(); // Add this

    //     // Prepare selected values
    //     $selectedSubjects = $writerProfile->subjects->pluck('id')->toArray();
    //     $selectedLanguages = $writerProfile->languages->pluck('id')->toArray();
    //     $selectedCitationStyles = $writerProfile->citationStyles->pluck('id')->toArray();

    //     return Inertia::render('Writer/Profile/Edit', [
    //         'user' => $user,
    //         'writerProfile' => array_merge($writerProfile->toArray(), [
    //             'subjects' => $selectedSubjects,
    //             'languages' => $selectedLanguages,
    //             'citation_styles' => $selectedCitationStyles,
    //         ]),
    //         'subjects' => $subjects,
    //         'languages' => $languages,
    //         'citationStyles' => $citationStyles,
    //         'educationLevels' => $educationLevels,
    //         'specializations' => $specializations,
    //         'writerLevels' => $writerLevels,
    //         'paymentMethods' => $paymentMethods, // Add this
    //         'isFullyQualified' => $isFullyQualified,
    //         'maxSubjects' => 5,
    //         'maxLanguages' => 3,
    //         'maxCitationStyles' => 5,
    //     ]);
    // }

    public function edit()
    {
        $user = Auth::user();
        $writerProfile = $user->writerProfile()->with([
            'subjects',
            'languages',
            'citationStyles',
            'paymentAccounts.paymentMethod'
        ])->first();

        $isFullyQualified = $writerProfile->isApproved();

        // Get qualification status
        $qualificationStatus = $writerProfile->getQualificationStatus();

        // Prepare data for form
        $subjects = Subject::select('id', 'name')->orderBy('name')->get();
        $languages = Language::select('id', 'name')->orderBy('name')->get();
        $citationStyles = WriterCitationStyle::select('id', 'name')->orderBy('name')->get();
        $educationLevels = EducationLevel::select('id', 'name')->orderBy('name')->get();
        $specializations = Specialization::select('id', 'name')->orderBy('name')->get();
        $writerLevels = WriterLevel::select('id', 'name')->orderBy('name')->get();
        $paymentMethods = WriterPaymentMethod::where('is_active', true)->get();

        return Inertia::render('Writer/Profile/Edit', [
            'profile' => $writerProfile, // Changed from 'writerProfile' to 'profile'
            'user' => $user,
            'qualificationStatus' => $qualificationStatus,
            'isFullyQualified' => $isFullyQualified,
            'educationLevels' => $educationLevels,
            'subjects' => $subjects,
            'languages' => $languages,
            'citationStyles' => $citationStyles,
            'writerLevels' => $writerLevels,
            'specializations' => $specializations,
            'paymentMethods' => $paymentMethods,
            'maxSubjects' => 5,
            'maxLanguages' => 3,
            'maxCitationStyles' => 5,
        ]);
    }

    /**
     * Update the writer profile.
     */
    public function update(Request $request)
    {
        $user = Auth::user();
        $writerProfile = $user->writerProfile;
        $isFullyQualified = $writerProfile->isApproved();

        // Basic validation for all users
        $rules = [
            'bio' => 'required|string|max:1000',
            'experience_years' => 'required|integer|min:0|max:50',
        ];

        // Additional validation for qualified writers
        if ($isFullyQualified) {
            $additionalRules = [
                'education_level_id' => 'required|exists:writer_education_levels,id',
                'specialization_id' => 'required|exists:writer_specializations,id',
                'writer_level_id' => 'nullable|exists:writer_levels,id',
                'subjects' => 'required|array|min:1|max:5',
                'subjects.*' => 'exists:subjects,id',
                'languages' => 'required|array|min:1|max:3',
                'languages.*' => 'exists:languages,id',
                'citation_styles' => 'required|array|min:1|max:5',
                'citation_styles.*' => 'exists:writer_citation_styles,id',
                'hourly_rate' => 'nullable|numeric|min:0|max:200',
                'is_available' => 'boolean',
                // Payment account fields
                'payment_method_id' => 'nullable|exists:writer_payment_methods,id',
                'payment_account_details' => 'nullable|string|max:1000',
            ];

            $rules = array_merge($rules, $additionalRules);
        }

        $validated = $request->validate($rules);

        // Update basic info for all users
        $writerProfile->bio = $validated['bio'];
        $writerProfile->experience_years = $validated['experience_years'];

        // Update additional info for qualified writers
        if ($isFullyQualified) {
            $writerProfile->education_level_id = $validated['education_level_id'];
            $writerProfile->specialization_id = $validated['specialization_id'];
            $writerProfile->writer_level_id = $validated['writer_level_id'] ?? null;
            $writerProfile->is_available = $validated['is_available'] ?? false;
            $writerProfile->hourly_rate = $validated['hourly_rate'] ?? null;

            // Update relationships
            $writerProfile->subjects()->sync($validated['subjects']);
            $writerProfile->languages()->sync($validated['languages']);
            $writerProfile->citationStyles()->sync($validated['citation_styles']);

            // Handle payment account
            if (!empty($validated['payment_method_id']) && !empty($validated['payment_account_details'])) {
                // Remove existing primary payment account
                $writerProfile->paymentAccounts()->update(['is_primary' => false]);

                // Create or update payment account
                $writerProfile->paymentAccounts()->updateOrCreate(
                    [
                        'payment_method_id' => $validated['payment_method_id'],
                    ],
                    [
                        'account_details' => $validated['payment_account_details'],
                        'is_primary' => true,
                        'is_verified' => false, // Admin needs to verify
                    ]
                );
            }
        }

        // Save changes
        $writerProfile->save();

        // Notify user of successful update
        $user->notify(new ProfileUpdated());

        return redirect()->route('writer.profile.edit')->with('success', 'Profile updated successfully.');
    }

    /**
     * Upload profile image.
     */
    public function uploadProfileImage(Request $request)
    {
        $request->validate([
            'profile_image' => 'required|image|mimes:jpeg,png,jpg,gif|max:2048',
        ]);

        $user = Auth::user();
        $writerProfile = $user->writerProfile;

        // Delete old image if exists
        if ($writerProfile->profile_image) {
            Storage::disk('public')->delete($writerProfile->profile_image);
        }

        // Store new image
        $path = $request->file('profile_image')->store('profile-images', 'public');

        // Update profile
        $writerProfile->profile_image = $path;
        $writerProfile->save();

        return response()->json([
            'success' => true,
            'message' => 'Profile image uploaded successfully',
            'image_url' => Storage::disk('public')->url($path)
        ]);
    }
}
