<?php

namespace App\Http\Controllers\Writer;

use App\Http\Controllers\Controller;
use App\Models\OrderRevision;
use App\Models\Order;
use Illuminate\Http\Request;
use Inertia\Inertia;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\DB;

class WriterRevisionController extends Controller
{
    public function __construct()
    {
        $this->middleware('auth');
        $this->middleware('writer');
    }

    /**
     * Display a listing of the writer's assigned revisions.
     */
    public function index(Request $request)
    {
        $writer = Auth::user();
        
        $revisions = OrderRevision::with(['order', 'order.client'])
            ->whereHas('order', function ($query) use ($writer) {
                $query->where('writer_id', $writer->id);
            })
            ->when($request->search, function ($query, $search) {
                $query->whereHas('order', function ($q) use ($search) {
                    $q->where('title', 'like', "%{$search}%")
                      ->orWhere('order_number', 'like', "%{$search}%");
                });
            })
            ->when($request->status, function ($query, $status) {
                $query->where('status', $status);
            })
            ->orderBy('created_at', 'desc')
            ->paginate(15);

        return Inertia::render('Writer/Revisions/Index', [
            'revisions' => $revisions,
            'filters' => $request->only(['search', 'status']),
            'statuses' => config('revisions.statuses'),
        ]);
    }

    /**
     * Display the writer's revision dashboard with statistics.
     */
    public function dashboard()
    {
        $writer = Auth::user();
        
        $stats = [
            'total_revisions' => OrderRevision::whereHas('order', function ($query) use ($writer) {
                $query->where('writer_id', $writer->id);
            })->count(),
            
            'pending_revisions' => OrderRevision::whereHas('order', function ($query) use ($writer) {
                $query->where('writer_id', $writer->id);
            })->where('status', 'requested')->count(),
            
            'in_progress_revisions' => OrderRevision::whereHas('order', function ($query) use ($writer) {
                $query->where('writer_id', $writer->id);
            })->where('status', 'in_progress')->count(),
            
            'completed_revisions' => OrderRevision::whereHas('order', function ($query) use ($writer) {
                $query->where('writer_id', $writer->id);
            })->where('status', 'completed')->count(),
            
            'rejected_revisions' => OrderRevision::whereHas('order', function ($query) use ($writer) {
                $query->where('writer_id', $writer->id);
            })->where('status', 'rejected')->count(),
        ];

        $recent_revisions = OrderRevision::with(['order', 'order.client'])
            ->whereHas('order', function ($query) use ($writer) {
                $query->where('writer_id', $writer->id);
            })
            ->orderBy('updated_at', 'desc')
            ->limit(5)
            ->get();

        return Inertia::render('Writer/Revisions/Dashboard', [
            'stats' => $stats,
            'recent_revisions' => $recent_revisions,
        ]);
    }

    /**
     * Display the specified revision.
     */
    public function show(OrderRevision $revision)
    {
        $writer = Auth::user();
        
        // Ensure the writer can only view revisions for their orders
        if ($revision->order->writer_id !== $writer->id) {
            abort(403, 'Unauthorized access to this revision.');
        }

        $revision->load(['order', 'order.client', 'conversation.messages.user']);

        return Inertia::render('Writer/Revisions/Show', [
            'revision' => $revision,
            'order' => $revision->order,
            'conversation' => $revision->conversation,
        ]);
    }

    /**
     * Start working on a revision.
     */
    public function startWorking(OrderRevision $revision)
    {
        $writer = Auth::user();
        
        // Ensure the writer can only work on revisions for their orders
        if ($revision->order->writer_id !== $writer->id) {
            abort(403, 'Unauthorized access to this revision.');
        }

        // Check if revision can be started
        if ($revision->status !== 'requested') {
            return back()->with('error', 'This revision cannot be started at this time.');
        }

        $revision->update([
            'status' => 'in_progress',
            'started_at' => now(),
        ]);

        // Update order status if this is the first revision being worked on
        if ($revision->order->status === 'under_review') {
            $revision->order->update(['status' => 'in_progress']);
        }

        return back()->with('success', 'Revision work started successfully.');
    }

    /**
     * Mark a revision as complete.
     */
    public function markComplete(Request $request, OrderRevision $revision)
    {
        $writer = Auth::user();
        
        // Ensure the writer can only complete revisions for their orders
        if ($revision->order->writer_id !== $writer->id) {
            abort(403, 'Unauthorized access to this revision.');
        }

        // Check if revision can be completed
        if ($revision->status !== 'in_progress') {
            return back()->with('error', 'This revision cannot be completed at this time.');
        }

        $request->validate([
            'completion_notes' => 'nullable|string|max:1000',
        ]);

        $revision->update([
            'status' => 'completed',
            'completed_at' => now(),
            'completion_notes' => $request->completion_notes,
        ]);

        // Check if all revisions for this order are completed
        $pending_revisions = OrderRevision::where('order_id', $revision->order_id)
            ->whereIn('status', ['requested', 'in_progress'])
            ->count();

        if ($pending_revisions === 0) {
            $revision->order->update(['status' => 'delivered_to_client']);
        }

        return back()->with('success', 'Revision marked as complete successfully.');
    }
} 