<?php

namespace App\Http\Middleware;

use App\Models\WriterEssayTest;
use Closure;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Log;
use Symfony\Component\HttpFoundation\Response;

class CheckExpiredEssayTests
{
    /**
     * The probability (0-100) that this middleware will run on any given request.
     * This helps distribute the load of checking for expired tests.
     */
    const RUN_PROBABILITY = 10; // 10% chance to run on any request

    /**
     * The maximum number of expired tests to process in a single request.
     */
    const BATCH_SIZE = 5;

    /**
     * Handle an incoming request.
     *
     * @param  \Closure(\Illuminate\Http\Request): (\Symfony\Component\HttpFoundation\Response)  $next
     */
    public function handle(Request $request, Closure $next): Response
    {
        // Only run this middleware with a certain probability to reduce database load
        if (rand(1, 100) <= self::RUN_PROBABILITY) {
            $this->processExpiredTests();
        }

        return $next($request);
    }

    /**
     * Process expired essay tests.
     *
     * @return void
     */
    private function processExpiredTests(): void
    {
        try {
            // Find essay tests that have expired but haven't been processed
            $expiredTests = WriterEssayTest::where('essay_test_status', 'essay_pending')
                ->whereNotNull('expected_end_at')
                ->where('expected_end_at', '<', now())
                ->whereNull('auto_processed_at')
                ->limit(self::BATCH_SIZE)
                ->get();

            if ($expiredTests->isEmpty()) {
                return;
            }

            Log::info("Processing " . $expiredTests->count() . " expired essay tests");

            foreach ($expiredTests as $test) {
                if ($test->hasUploadedFile()) {
                    // Process partial submission if a file was uploaded
                    $test->processPartialSubmission();
                    Log::info("Auto-processed partial submission for essay test ID: {$test->id}");
                } else {
                    // Mark as expired with 3-month retry period if no file was uploaded
                    $test->markAsExpired('Auto-processed by middleware - no file uploaded');
                    Log::info("Marked essay test ID: {$test->id} as expired with 3-month retry period");
                }
            }
        } catch (\Exception $e) {
            // Log the error but don't interrupt the request
            Log::error("Error processing expired essay tests: " . $e->getMessage());
            Log::error($e->getTraceAsString());
        }
    }
}
