<?php

namespace App\Http\Middleware;

use Closure;
use Illuminate\Http\Request;
use Symfony\Component\HttpFoundation\Response;
use Illuminate\Support\Facades\Auth;

class ClientMiddleware
{
    public function handle(Request $request, Closure $next): Response
    {
        // Allow broadcasting routes for all authenticated users
        // Broadcasting needs to work across all user types for messaging
        if ($this->isBroadcastingRoute($request)) {
            if (Auth::check()) {
                return $next($request);
            }
            return redirect()->route('login');
        }

        if (Auth::check() && Auth::user()->user_type === 'client') {
            return $next($request);
        }

        if (Auth::check()) {
            // Check if we're already on a dashboard route to prevent redirect loops
            $currentRoute = $request->route()->getName();

            switch (Auth::user()->user_type) {
                case 'admin':
                case 'super_admin':
                    $targetRoute = 'admin.dashboard';
                    break;
                case 'writer':
                    $targetRoute = 'writer.dashboard';
                    break;
                default:
                    $targetRoute = 'login';
                    break;
            }

            // Only redirect if we're not already on the target route
            if ($currentRoute !== $targetRoute) {
                return redirect()->route($targetRoute);
            }

            // If we're already on the correct route, abort with 403 (Forbidden)
            abort(403, 'You do not have access to this section.');
        }

        return redirect()->route('login');
    }

    /**
     * Check if the current request is for a broadcasting route
     * Broadcasting routes need to be accessible to all authenticated users
     * for real-time messaging functionality
     */
    private function isBroadcastingRoute(Request $request): bool
    {
        $routeName = $request->route()?->getName();
        $path = $request->path();

        return $routeName === 'broadcasting.auth' || 
               str_starts_with($path, 'broadcasting/') ||
               str_starts_with($routeName ?? '', 'broadcasting.');
    }
}
