<?php

namespace App\Http\Middleware;

use App\Models\FooterSetting;
use Illuminate\Http\Request;
use Inertia\Middleware;
use Illuminate\Support\Facades\Cache;

class HandleInertiaRequests extends Middleware
{
    /**
     * The root template that is loaded on the first page visit.
     *
     * @var string
     */
    protected $rootView = 'app';

    /**
     * Determine the current asset version.
     */
    public function version(Request $request): ?string
    {
        return parent::version($request);
    }

    /**
     * Define the props that are shared by default.
     *
     * @return array<string, mixed>
     */
    public function share(Request $request): array
    {
        return [
            ...parent::share($request),
            'auth' => [
                'user' => $request->user() ? $request->user()->only(['id', 'name', 'email', 'user_type']) : null,
                'verified' => $request->user() ? $request->user()->hasVerifiedEmail() : null,
            ],
            'flash' => [
                'message' => fn() => $request->session()->get('message'),
                'error' => fn() => $request->session()->get('error'),
            ],
            'footerData' => fn() => $this->getFooterData(),
        ];
    }

    /**
     * Get footer data organized by section
     */
    private function getFooterData(): array
    {
        return Cache::remember('footer_data', 3600, function () {
            $settings = FooterSetting::active()->ordered()->get()->groupBy('section_type');

            $footerData = [];

            // Company Info
            $companyInfo = $settings->get('company_info', collect());
            $mainInfo = $companyInfo->where('key', 'company_info')->first();
            $copyrightInfo = $companyInfo->where('key', 'copyright_text')->first();

            $footerData['companyInfo'] = [
                'title' => $mainInfo?->data['title'] ?? 'Academic Scribe',
                'description' => $mainInfo?->data['description'] ?? '',
            ];

            $footerData['copyrightText'] = $copyrightInfo?->data['text'] ?? 'Academic Scribe. All rights reserved.';

            // Service Links
            $footerData['serviceLinks'] = $settings->get('service_link', collect())
                ->map(function ($link) {
                    return [
                        'name' => $link->data['name'] ?? '',
                        'url' => $link->data['url'] ?? '#',
                    ];
                })
                ->values()
                ->toArray();

            // Quick Links
            $footerData['quickLinks'] = $settings->get('quick_link', collect())
                ->map(function ($link) {
                    return [
                        'name' => $link->data['name'] ?? '',
                        'url' => $link->data['url'] ?? '#',
                    ];
                })
                ->values()
                ->toArray();

            // Contact Info
            $contactInfo = $settings->get('contact_info', collect())->first();
            $footerData['contactInfo'] = [
                'email' => $contactInfo?->data['email'] ?? '',
                'phone' => $contactInfo?->data['phone'] ?? '',
                'phone_description' => $contactInfo?->data['phone_description'] ?? '',
                'address' => $contactInfo?->data['address'] ?? [],
            ];

            // Legal Links
            $footerData['legalLinks'] = $settings->get('legal_link', collect())
                ->map(function ($link) {
                    return [
                        'name' => $link->data['name'] ?? '',
                        'url' => $link->data['url'] ?? '#',
                        'external' => $link->data['external'] ?? false,
                    ];
                })
                ->values()
                ->toArray();

            return $footerData;
        });
    }
}
