<?php

namespace App\Http\Middleware;

use Closure;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\View;
use Inertia\Inertia;

class InjectMetaTagsForCrawlers
{
    /**
     * Known crawler user agents
     */
    protected $crawlers = [
        'facebookexternalhit',
        'facebot',
        'LinkedInBot',
        'Twitterbot',
        'Slackbot',
        'WhatsApp',
        'SkypeUriPreview',
        'Googlebot',
        'bingbot',
        'yahoo',
        'AhrefsBot',
        'SemrushBot',
        'DuckDuckBot',
        'Baiduspider',
        'YandexBot',
        'developers.google.com/+/web/snippet',
    ];

    /**
     * Handle an incoming request.
     */
    public function handle(Request $request, Closure $next)
    {
        // Check if this is a crawler
        $userAgent = $request->header('User-Agent', '');
        $isCrawler = $this->isCrawler($userAgent);

        if ($isCrawler) {
            // Force Inertia to NOT use client-side rendering for crawlers
            // This makes Inertia send JSON data in first response
            $request->headers->set('X-Inertia', 'true');
            $request->headers->set('X-Inertia-Version', Inertia::getVersion());
            
            // Get the Inertia response
            $response = $next($request);
            
            // Now check if response has Inertia data
            if ($response->headers->get('X-Inertia')) {
                // Get the Inertia page data
                $page = json_decode($response->getContent(), true);
                
                // Extract meta data from page props
                $metaData = $this->extractMetaData($page, $request);
                
                // Return a simple HTML response with meta tags for crawlers
                return response()->view('crawler', [
                    'meta' => $metaData,
                    'url' => $request->url(),
                ]);
            }
        }

        return $next($request);
    }

    /**
     * Check if the user agent is a crawler
     */
    protected function isCrawler(string $userAgent): bool
    {
        $userAgent = strtolower($userAgent);
        
        foreach ($this->crawlers as $crawler) {
            if (str_contains($userAgent, strtolower($crawler))) {
                return true;
            }
        }
        
        return false;
    }

    /**
     * Extract meta data from Inertia page props
     */
    protected function extractMetaData(array $page, Request $request): array
    {
        $props = $page['props'] ?? [];
        $component = $page['component'] ?? '';
        
        // Check for pageData in props (CMS pages)
        $pageData = $props['pageData'] ?? [];
        
        // Default meta tags
        $meta = [
            'title' => $pageData['title'] ?? config('app.name') . ' - Professional Academic Writing Services',
            'description' => $pageData['description'] ?? 'Get professional academic writing help from expert writers. Essays, research papers, dissertations and more. 100% original, plagiarism-free work guaranteed.',
            'keywords' => $pageData['keywords'] ?? 'academic writing, essay writing service, research papers, dissertation help',
            'canonical' => $pageData['canonicalUrl'] ?? $request->url(),
            'og_title' => $pageData['ogTitle'] ?? $pageData['title'] ?? config('app.name'),
            'og_description' => $pageData['ogDescription'] ?? $pageData['description'] ?? '',
            'og_image' => $pageData['ogImage'] ?? asset('images/og-default.jpg'),
            'og_url' => $pageData['ogUrl'] ?? $request->url(),
            'og_type' => $pageData['ogType'] ?? 'website',
            'twitter_card' => $pageData['twitterCard'] ?? 'summary_large_image',
            'twitter_title' => $pageData['twitterTitle'] ?? $pageData['title'] ?? '',
            'twitter_description' => $pageData['twitterDescription'] ?? $pageData['description'] ?? '',
            'twitter_image' => $pageData['twitterImage'] ?? asset('images/og-default.jpg'),
            'structured_data' => $pageData['structuredData'] ?? null,
        ];

        // Component-specific meta tags
        $meta = $this->getComponentMeta($component, $props, $meta, $request);

        return $meta;
    }

    /**
     * Get component-specific meta tags
     */
    protected function getComponentMeta(string $component, array $props, array $meta, Request $request): array
    {
        // Homepage
        if ($component === 'Welcome') {
            $meta['title'] = 'Professional Academic Writing Services | 100% Original Content & AI-Free';
            $meta['description'] = 'Get expert academic writing help from qualified writers. Essays, research papers, dissertations & more. Plagiarism-free guarantee, 24/7 support, on-time delivery.';
        }

        // Service pages
        if (str_starts_with($component, 'Services/')) {
            $service = $props['service'] ?? [];
            if (!empty($service)) {
                $meta['title'] = $service['meta_title'] ?? $service['title'] ?? $meta['title'];
                $meta['description'] = $service['meta_description'] ?? $meta['description'];
                
                // Extract structured data from database if available
                if (!empty($service['structured_data'])) {
                    $meta['structured_data'] = is_string($service['structured_data']) 
                        ? json_decode($service['structured_data'], true) 
                        : $service['structured_data'];
                }
            }
        }

        // Blog posts
        if ($component === 'Blog/BlogPost') {
            $post = $props['post'] ?? [];
            if (!empty($post)) {
                $meta['title'] = $post['meta_title'] ?? $post['title'] ?? $meta['title'];
                $meta['description'] = $post['meta_description'] ?? $post['excerpt'] ?? $meta['description'];
                $meta['og_image'] = $post['featured_image'] ?? $meta['og_image'];
                $meta['og_type'] = 'article';
                
                // Extract structured data from database if available
                if (!empty($post['structured_data'])) {
                    $meta['structured_data'] = is_string($post['structured_data']) 
                        ? json_decode($post['structured_data'], true) 
                        : $post['structured_data'];
                }
            }
        }

        // About page
        if ($component === 'About') {
            $seoData = $props['seoData'] ?? [];
            if (!empty($seoData)) {
                $meta['title'] = $seoData['title'] ?? $meta['title'];
                $meta['description'] = $seoData['description'] ?? $meta['description'];
            }
        }

        return $meta;
    }
}
