<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;

class Bid extends Model
{
    use HasFactory;

    protected $fillable = [
        'order_id',
        'writer_id',
        'bid_amount',
        'completion_time',
        'comments',
        'status',
    ];

    protected $with = ['writer'];

    public function order()
    {
        return $this->belongsTo(Order::class);
    }

    /**
     * Get the writer who placed the bid.
     */
    public function writer()
    {
        return $this->belongsTo(User::class, 'writer_id')->select(['id', 'name', 'email']);
    }

    /**
     * Get safe writer information for client-facing views.
     *
     * @return array
     */
    public function getSafeWriterInfo()
    {
        $nickname = $this->writer && $this->writer->writerProfile && $this->writer->writerProfile->nickname
            ? $this->writer->writerProfile->nickname
            : "Writer #" . $this->writer_id;

        $experience = $this->writer && $this->writer->writerProfile
            ? $this->writer->writerProfile->experience_years
            : null;

        return [
            'nickname' => $nickname,
            'experience_years' => $experience,
        ];
    }

    /**
     * Scope a query to filter bids by status.
     *
     * @param \Illuminate\Database\Eloquent\Builder $query
     * @param string $status
     * @return \Illuminate\Database\Eloquent\Builder
     */
    public function scopeStatus($query, $status)
    {
        if ($status && $status !== 'all') {
            return $query->where('status', $status);
        }

        return $query;
    }

    /**
     * Scope a query to filter bids by date range.
     *
     * @param \Illuminate\Database\Eloquent\Builder $query
     * @param string|null $fromDate
     * @param string|null $toDate
     * @return \Illuminate\Database\Eloquent\Builder
     */
    public function scopeDateRange($query, $fromDate = null, $toDate = null)
    {
        if ($fromDate) {
            $query->whereDate('created_at', '>=', $fromDate);
        }

        if ($toDate) {
            $query->whereDate('created_at', '<=', $toDate);
        }

        return $query;
    }

    /**
     * Scope a query to search bids by order information or writer.
     *
     * @param \Illuminate\Database\Eloquent\Builder $query
     * @param string|null $search
     * @return \Illuminate\Database\Eloquent\Builder
     */
    public function scopeSearch($query, $search = null)
    {
        if ($search) {
            return $query->where(function ($q) use ($search) {
                $q->whereHas('order', function ($q) use ($search) {
                    $q->where('order_number', 'like', "%{$search}%")
                        ->orWhere('title', 'like', "%{$search}%");
                })
                    ->orWhereHas('writer', function ($q) use ($search) {
                        $q->where('name', 'like', "%{$search}%");
                    })
                    ->orWhereHas('writer.writerProfile', function ($q) use ($search) {
                        $q->where('nickname', 'like', "%{$search}%");
                    });
            });
        }

        return $query;
    }

    /**
     * Scope a query to sort bids.
     *
     * @param \Illuminate\Database\Eloquent\Builder $query
     * @param string $field
     * @param string $direction
     * @return \Illuminate\Database\Eloquent\Builder
     */
    public function scopeSort($query, $field = 'created_at', $direction = 'desc')
    {
        $allowedFields = ['created_at', 'bid_amount', 'completion_time', 'status'];

        if (!in_array($field, $allowedFields)) {
            $field = 'created_at';
        }

        $direction = strtolower($direction) === 'asc' ? 'asc' : 'desc';

        return $query->orderBy($field, $direction);
    }

    /**
     * Get pending bids.
     *
     * @param \Illuminate\Database\Eloquent\Builder $query
     * @return \Illuminate\Database\Eloquent\Builder
     */
    public function scopePending($query)
    {
        return $query->where('status', 'pending');
    }

    /**
     * Get accepted bids.
     *
     * @param \Illuminate\Database\Eloquent\Builder $query
     * @return \Illuminate\Database\Eloquent\Builder
     */
    public function scopeAccepted($query)
    {
        return $query->where('status', 'accepted');
    }

    /**
     * Get rejected bids.
     *
     * @param \Illuminate\Database\Eloquent\Builder $query
     * @return \Illuminate\Database\Eloquent\Builder
     */
    public function scopeRejected($query)
    {
        return $query->where('status', 'rejected');
    }
}
