<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Relations\HasMany;
use Illuminate\Database\Eloquent\Relations\BelongsToMany;
use Illuminate\Database\Eloquent\Relations\BelongsTo;
use App\Events\ConversationStatusChanged;
use App\Notifications\ConversationStatusNotification;

class Conversation extends Model
{
    use HasFactory;

    protected $fillable = [
        'conversation_type',
        'title',
        'status',
        'resolved_at',
        'resolved_by',
        'archived_at',
        'archived_by',
        'last_activity_at',
        'priority_level',
    ];

    protected $casts = [
        'created_at' => 'datetime',
        'updated_at' => 'datetime',
        'resolved_at' => 'datetime',
        'archived_at' => 'datetime',
        'last_activity_at' => 'datetime',
    ];

    /**
     * Get the participants for this conversation.
     */
    public function participants(): BelongsToMany
    {
        return $this->belongsToMany(User::class, 'conversation_participants')
                    ->withPivot('user_type', 'joined_at', 'left_at')
                    ->withTimestamps();
    }

    /**
     * Get the messages for this conversation.
     */
    public function messages(): HasMany
    {
        return $this->hasMany(Message::class)->orderBy('created_at', 'asc');
    }

    /**
     * Get the last message for this conversation.
     */
    public function lastMessage()
    {
        return $this->hasOne(Message::class)->latest();
    }

    /**
     * Get unread messages count for a specific user.
     */
    public function getUnreadCountForUser(User $user): int
    {
        return $this->messages()
                    ->whereDoesntHave('reads', function ($query) use ($user) {
                        $query->where('user_id', $user->id);
                    })
                    ->where('sender_id', '!=', $user->id)
                    ->count();
    }

    /**
     * Check if a user is a participant in this conversation.
     */
    public function hasParticipant(User $user): bool
    {
        return $this->participants()->where('user_id', $user->id)->exists();
    }

    /**
     * Get the other participant in a two-person conversation.
     */
    public function getOtherParticipant(User $user)
    {
        return $this->participants()
                    ->where('user_id', '!=', $user->id)
                    ->first();
    }

    /**
     * Scope for conversations involving a specific user.
     */
    public function scopeForUser($query, User $user)
    {
        return $query->whereHas('participants', function ($q) use ($user) {
            $q->where('user_id', $user->id);
        });
    }

    /**
     * Scope for conversations of a specific type.
     */
    public function scopeOfType($query, string $type)
    {
        return $query->where('conversation_type', $type);
    }

    /**
     * Get the user who resolved this conversation.
     */
    public function resolvedBy(): BelongsTo
    {
        return $this->belongsTo(User::class, 'resolved_by');
    }

    /**
     * Get the user who archived this conversation.
     */
    public function archivedBy(): BelongsTo
    {
        return $this->belongsTo(User::class, 'archived_by');
    }

    /**
     * Scope for active conversations.
     */
    public function scopeActive($query)
    {
        return $query->where('status', 'active');
    }

    /**
     * Scope for resolved conversations.
     */
    public function scopeResolved($query)
    {
        return $query->where('status', 'resolved');
    }

    /**
     * Scope for archived conversations.
     */
    public function scopeArchived($query)
    {
        return $query->where('status', 'archived');
    }

    /**
     * Scope for escalated conversations.
     */
    public function scopeEscalated($query)
    {
        return $query->where('status', 'escalated');
    }

    /**
     * Scope for non-archived conversations (active, resolved, escalated).
     */
    public function scopeNotArchived($query)
    {
        return $query->whereIn('status', ['active', 'resolved', 'escalated']);
    }

    /**
     * Mark conversation as resolved.
     */
    public function markAsResolved(User $user, bool $isAutomatic = false): bool
    {
        $oldStatus = $this->status;
        
        $result = $this->update([
            'status' => 'resolved',
            'resolved_at' => now(),
            'resolved_by' => $user->id,
        ]);
        
        if ($result) {
            $this->fireStatusChangeEvent($oldStatus, 'resolved', $user, $isAutomatic);
        }
        
        return $result;
    }

    /**
     * Reopen a resolved conversation.
     */
    public function reopen(User $user = null): bool
    {
        $oldStatus = $this->status;
        
        $result = $this->update([
            'status' => 'active',
            'resolved_at' => null,
            'resolved_by' => null,
        ]);
        
        if ($result && $user) {
            $this->fireStatusChangeEvent($oldStatus, 'active', $user);
        }
        
        return $result;
    }

    /**
     * Archive conversation (admin only).
     */
    public function archive(User $user, bool $isAutomatic = false): bool
    {
        $oldStatus = $this->status;
        
        $result = $this->update([
            'status' => 'archived',
            'archived_at' => now(),
            'archived_by' => $user->id,
        ]);
        
        if ($result) {
            $this->fireStatusChangeEvent($oldStatus, 'archived', $user, $isAutomatic);
        }
        
        return $result;
    }

    /**
     * Escalate conversation (admin only).
     */
    public function escalate(User $user, int $priorityLevel = 5): bool
    {
        $oldStatus = $this->status;
        
        $result = $this->update([
            'status' => 'escalated',
            'priority_level' => $priorityLevel,
        ]);
        
        if ($result) {
            $this->fireStatusChangeEvent($oldStatus, 'escalated', $user);
        }
        
        return $result;
    }

    /**
     * Update last activity timestamp.
     */
    public function updateLastActivity(): bool
    {
        return $this->update([
            'last_activity_at' => now(),
        ]);
    }

    /**
     * Check if conversation can be reopened by user.
     */
    public function canBeReopenedBy(User $user): bool
    {
        // Must be resolved
        if ($this->status !== 'resolved') {
            return false;
        }

        // Admins can always reopen
        if (in_array($user->user_type, ['admin', 'super_admin'])) {
            return true;
        }

        // Clients/writers can reopen within 48 hours
        if ($this->resolved_at && $this->resolved_at->gt(now()->subHours(48))) {
            return $this->hasParticipant($user);
        }

        return false;
    }

    /**
     * Check if conversation can be archived by user.
     */
    public function canBeArchivedBy(User $user): bool
    {
        return in_array($user->user_type, ['admin', 'super_admin']);
    }

    /**
     * Check if conversation can be resolved by user.
     */
    public function canBeResolvedBy(User $user): bool
    {
        return $this->status === 'active' && $this->hasParticipant($user);
    }

    /**
     * Get status display name.
     */
    public function getStatusDisplayAttribute(): string
    {
        return match($this->status) {
            'active' => 'Active',
            'resolved' => 'Resolved',
            'archived' => 'Archived',
            'escalated' => 'Escalated',
            default => 'Unknown',
        };
    }

    /**
     * Get status color for UI.
     */
    public function getStatusColorAttribute(): string
    {
        return match($this->status) {
            'active' => 'green',
            'resolved' => 'gray',
            'archived' => 'gray',
            'escalated' => 'red',
            default => 'gray',
        };
    }

    /**
     * Fire status change event and send notifications.
     */
    private function fireStatusChangeEvent(string $oldStatus, string $newStatus, User $changedBy, bool $isAutomatic = false): void
    {
        // Fire the event for real-time updates
        event(new ConversationStatusChanged($this, $oldStatus, $newStatus, $changedBy, $isAutomatic));
        
        // Send notifications to all participants except the one who made the change
        $participants = $this->participants()->where('user_id', '!=', $changedBy->id)->get();
        
        foreach ($participants as $participant) {
            $participant->notify(new ConversationStatusNotification(
                $this,
                $oldStatus,
                $newStatus,
                $changedBy,
                $isAutomatic
            ));
        }
    }
} 