<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;

class OrderFile extends Model
{
    use HasFactory;

    protected $table = 'order_files';

    public $timestamps = false; // Disable timestamps

    // File category constants
    const CATEGORY_SAMPLE = 'sample';
    const CATEGORY_INSTRUCTIONS = 'instructions';
    const CATEGORY_DRAFT = 'draft';
    const CATEGORY_COMPLETE = 'complete';
    const CATEGORY_ADMIN = 'admin';
    const CATEGORY_GENERAL = 'general';

    // Access control constants
    const ACCESS_ADMIN_ONLY = 'admin_only';
    const ACCESS_CLIENT_WRITER = 'client_writer';
    const ACCESS_CLIENT_ADMIN = 'client_admin';
    const ACCESS_WRITER_ADMIN = 'writer_admin';

    // File status constants
    const STATUS_PENDING = 'pending';
    const STATUS_APPROVED = 'approved';
    const STATUS_REJECTED = 'rejected';

    protected $fillable = [
        'order_id',
        'file_name',
        'file_type',
        'uploader_id',
        'uploader_name',
        'uploader_user_type',
        'file_url',
        'file_size',
        'file_status',
        'file_category',
        'description',
        'access_control',
        'date_uploaded'
    ];

    protected $casts = [
        'date_uploaded' => 'datetime',
    ];

    // Define relationships
    public function order()
    {
        return $this->belongsTo(Order::class, 'order_id');
    }

    public function uploader()
    {
        return $this->belongsTo(User::class, 'uploader_id');
    }

    // Scopes for filtering files by category
    public function scopeByCategory($query, $category)
    {
        return $query->where('file_category', $category);
    }

    public function scopeByUploaderType($query, $userType)
    {
        return $query->where('uploader_user_type', $userType);
    }

    // Helper methods for access control
    public function canBeAccessedBy($userType)
    {
        switch ($this->access_control) {
            case self::ACCESS_ADMIN_ONLY:
                return $userType === 'admin';
            case self::ACCESS_CLIENT_WRITER:
                return in_array($userType, ['client', 'writer']);
            case self::ACCESS_CLIENT_ADMIN:
                return in_array($userType, ['client', 'admin']);
            case self::ACCESS_WRITER_ADMIN:
                return in_array($userType, ['writer', 'admin']);
            default:
                return false;
        }
    }

    // Check if file is client upload
    public function isClientUpload()
    {
        return $this->uploader_user_type === 'client';
    }

    // Check if file is writer upload
    public function isWriterUpload()
    {
        return $this->uploader_user_type === 'writer';
    }

    // Check if file is admin upload
    public function isAdminUpload()
    {
        return $this->uploader_user_type === 'admin';
    }

    // Accessor for file_path (compatibility with existing code)
    public function getFilePathAttribute()
    {
        return $this->file_url;
    }

    // Accessor for uploaded_by (compatibility with existing code)
    public function getUploadedByAttribute()
    {
        return $this->uploader_id;
    }

    // Get the full file path for storage
    public function getFullPathAttribute()
    {
        return storage_path('app/public/' . $this->file_url);
    }

    // Check if file exists in storage
    public function exists()
    {
        return file_exists($this->full_path);
    }
}
