<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Relations\BelongsTo;
use Illuminate\Database\Eloquent\Relations\HasMany;

class OrderRevision extends Model
{
    use HasFactory;

    // Revision type constants
    const REVISION_TYPE_CONTENT = 'content';
    const REVISION_TYPE_FORMATTING = 'formatting';
    const REVISION_TYPE_RESEARCH = 'research';
    const REVISION_TYPE_CITATION = 'citation';

    // Revision priority constants
    const PRIORITY_LOW = 'low';
    const PRIORITY_MEDIUM = 'medium';
    const PRIORITY_HIGH = 'high';

    // Revision status constants
    const STATUS_REQUESTED = 'requested';
    const STATUS_ACKNOWLEDGED = 'acknowledged';
    const STATUS_IN_PROGRESS = 'in_progress';
    const STATUS_SUBMITTED = 'submitted';
    const STATUS_REVIEWED = 'reviewed';

    protected $fillable = [
        'order_id',
        'requested_by',
        'assigned_to',
        'revision_type',
        'revision_priority',
        'revision_section',
        'revision_reason',
        'specific_instructions',
        'status',
        'revision_status',
        'requested_at',
        'revision_deadline',
        'started_at',
        'completed_at',
        'rejected_at',
        'quality_rating',
        'quality_notes',
        'payment_delay_hours',
        'quality_penalty',
        'quality_bonus',
        'revision_number',
        'is_final_revision',
        'conversation_id',
        'admin_notes',
        'escalation_reason',
        'revision_metadata',
        'revision_acknowledged_at',
        'revision_submitted_at',
        'revision_reviewed_at',
    ];

    protected $casts = [
        'requested_at' => 'datetime',
        'revision_deadline' => 'datetime',
        'started_at' => 'datetime',
        'completed_at' => 'datetime',
        'rejected_at' => 'datetime',
        'revision_acknowledged_at' => 'datetime',
        'revision_submitted_at' => 'datetime',
        'revision_reviewed_at' => 'datetime',
        'is_final_revision' => 'boolean',
        'payment_delay_hours' => 'decimal:2',
        'quality_penalty' => 'decimal:2',
        'quality_bonus' => 'decimal:2',
        'revision_metadata' => 'array',
    ];

    /**
     * Get the order that this revision belongs to.
     */
    public function order(): BelongsTo
    {
        return $this->belongsTo(Order::class);
    }

    /**
     * Get the admin who requested the revision.
     */
    public function requestedBy(): BelongsTo
    {
        return $this->belongsTo(User::class, 'requested_by');
    }

    /**
     * Get the writer assigned to complete the revision.
     */
    public function assignedTo(): BelongsTo
    {
        return $this->belongsTo(User::class, 'assigned_to');
    }

    /**
     * Get the conversation thread for this revision.
     */
    public function conversation(): BelongsTo
    {
        return $this->belongsTo(Conversation::class);
    }

    /**
     * Check if the revision is overdue.
     */
    public function isOverdue(): bool
    {
        if (!$this->revision_deadline || $this->revision_status === self::STATUS_REVIEWED) {
            return false;
        }
        
        return now()->isAfter($this->revision_deadline);
    }

    /**
     * Get the overdue hours if applicable.
     */
    public function getOverdueHours(): float
    {
        if (!$this->isOverdue()) {
            return 0;
        }
        
        return now()->diffInHours($this->revision_deadline);
    }

    /**
     * Calculate the total payment impact of this revision.
     */
    public function getTotalPaymentImpact(): float
    {
        $impact = 0;
        
        // Add quality penalties
        if ($this->quality_penalty > 0) {
            $impact += $this->quality_penalty;
        }
        
        // Subtract quality bonuses
        if ($this->quality_bonus > 0) {
            $impact -= $this->quality_bonus;
        }
        
        return $impact;
    }

    /**
     * Check if this revision can be escalated.
     */
    public function canBeEscalated(): bool
    {
        return in_array($this->revision_status, [self::STATUS_REQUESTED, self::STATUS_IN_PROGRESS]) && 
               $this->isOverdue() && 
               $this->getOverdueHours() >= 24; // 24 hours overdue
    }

    /**
     * Check if revision can transition to a new status.
     */
    public function canTransitionTo(string $newStatus): bool
    {
        $allowedTransitions = [
            self::STATUS_REQUESTED => [self::STATUS_ACKNOWLEDGED, self::STATUS_IN_PROGRESS],
            self::STATUS_ACKNOWLEDGED => [self::STATUS_IN_PROGRESS],
            self::STATUS_IN_PROGRESS => [self::STATUS_SUBMITTED],
            self::STATUS_SUBMITTED => [self::STATUS_REVIEWED],
            self::STATUS_REVIEWED => [], // Final state
        ];

        return in_array($newStatus, $allowedTransitions[$this->revision_status] ?? []);
    }

    /**
     * Get next possible statuses for this revision.
     */
    public function getNextPossibleStatuses(): array
    {
        $allowedTransitions = [
            self::STATUS_REQUESTED => [self::STATUS_ACKNOWLEDGED, self::STATUS_IN_PROGRESS],
            self::STATUS_ACKNOWLEDGED => [self::STATUS_IN_PROGRESS],
            self::STATUS_IN_PROGRESS => [self::STATUS_SUBMITTED],
            self::STATUS_SUBMITTED => [self::STATUS_REVIEWED],
            self::STATUS_REVIEWED => [], // Final state
        ];

        return $allowedTransitions[$this->revision_status] ?? [];
    }

    /**
     * Calculate revision deadline based on priority.
     */
    public function calculateDeadline(): \Carbon\Carbon
    {
        $baseHours = match($this->revision_priority) {
            self::PRIORITY_HIGH => 4,
            self::PRIORITY_MEDIUM => 8,
            self::PRIORITY_LOW => 24,
            default => 8,
        };

        return now()->addHours($baseHours);
    }

    /**
     * Get revision priority display name.
     */
    public function getPriorityDisplayName(): string
    {
        return match($this->revision_priority) {
            self::PRIORITY_HIGH => 'High Priority',
            self::PRIORITY_MEDIUM => 'Medium Priority',
            self::PRIORITY_LOW => 'Low Priority',
            default => 'Unknown Priority',
        };
    }

    /**
     * Get revision type display name.
     */
    public function getTypeDisplayName(): string
    {
        return match($this->revision_type) {
            self::REVISION_TYPE_CONTENT => 'Content Revision',
            self::REVISION_TYPE_FORMATTING => 'Formatting Revision',
            self::REVISION_TYPE_RESEARCH => 'Research Revision',
            self::REVISION_TYPE_CITATION => 'Citation Revision',
            default => 'Unknown Type',
        };
    }

    /**
     * Scope for active revisions.
     */
    public function scopeActive($query)
    {
        return $query->whereIn('revision_status', [self::STATUS_REQUESTED, self::STATUS_ACKNOWLEDGED, self::STATUS_IN_PROGRESS]);
    }

    /**
     * Scope for overdue revisions.
     */
    public function scopeOverdue($query)
    {
        return $query->where('revision_deadline', '<', now())
                    ->whereIn('revision_status', [self::STATUS_REQUESTED, self::STATUS_ACKNOWLEDGED, self::STATUS_IN_PROGRESS]);
    }

    /**
     * Scope for completed revisions.
     */
    public function scopeCompleted($query)
    {
        return $query->where('revision_status', self::STATUS_REVIEWED);
    }

    /**
     * Scope for revisions by type.
     */
    public function scopeByType($query, $type)
    {
        return $query->where('revision_type', $type);
    }

    /**
     * Scope for revisions by priority.
     */
    public function scopeByPriority($query, $priority)
    {
        return $query->where('revision_priority', $priority);
    }

    /**
     * Scope for revisions by writer.
     */
    public function scopeByWriter($query, $writerId)
    {
        return $query->where('assigned_to', $writerId);
    }

    /**
     * Scope for high priority revisions.
     */
    public function scopeHighPriority($query)
    {
        return $query->where('revision_priority', self::PRIORITY_HIGH);
    }

    /**
     * Scope for revisions by status.
     */
    public function scopeByStatus($query, $status)
    {
        return $query->where('revision_status', $status);
    }
} 