<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Factories\HasFactory;

class OrderStatusHistory extends Model
{
    use HasFactory;

    /**
     * The table associated with the model.
     *
     * @var string
     */
    protected $table = 'order_status_history';

    protected $fillable = [
        'order_id',
        'from_status',
        'to_status',
        'changed_by',
        'reason',
        'metadata',
    ];

    protected $casts = [
        'metadata' => 'array',
    ];

    /**
     * Get the order that owns the status history.
     */
    public function order()
    {
        return $this->belongsTo(Order::class);
    }

    /**
     * Get the user who made the status change.
     */
    public function changedByUser()
    {
        return $this->belongsTo(User::class, 'changed_by');
    }

    /**
     * Get the human-readable status name.
     */
    public function getFromStatusDisplayName(): string
    {
        $statusNames = [
            Order::ORDER_STATUS_PENDING => 'Pending',
            Order::ORDER_STATUS_BIDDING => 'Bidding',
            Order::ORDER_STATUS_ASSIGNED => 'Assigned',
            Order::ORDER_STATUS_IN_PROGRESS => 'In Progress',
            Order::ORDER_STATUS_SUBMITTED => 'Submitted',
            Order::ORDER_STATUS_UNDER_REVIEW => 'Under Review',
            Order::ORDER_STATUS_APPROVED => 'Approved',
            Order::ORDER_STATUS_REVISION_REQUESTED => 'Revision Requested',
            Order::ORDER_STATUS_CANCELLED => 'Cancelled',
        ];

        return $statusNames[$this->from_status] ?? 'Unknown';
    }

    /**
     * Get the human-readable status name.
     */
    public function getToStatusDisplayName(): string
    {
        $statusNames = [
            Order::ORDER_STATUS_PENDING => 'Pending',
            Order::ORDER_STATUS_BIDDING => 'Bidding',
            Order::ORDER_STATUS_ASSIGNED => 'Assigned',
            Order::ORDER_STATUS_IN_PROGRESS => 'In Progress',
            Order::ORDER_STATUS_SUBMITTED => 'Submitted',
            Order::ORDER_STATUS_UNDER_REVIEW => 'Under Review',
            Order::ORDER_STATUS_APPROVED => 'Approved',
            Order::ORDER_STATUS_REVISION_REQUESTED => 'Revision Requested',
            Order::ORDER_STATUS_CANCELLED => 'Cancelled',
        ];

        return $statusNames[$this->to_status] ?? 'Unknown';
    }

    /**
     * Get the status change description.
     */
    public function getStatusChangeDescription(): string
    {
        if ($this->from_status === null) {
            return "Order created with status: {$this->getToStatusDisplayName()}";
        }

        return "Status changed from {$this->getFromStatusDisplayName()} to {$this->getToStatusDisplayName()}";
    }

    /**
     * Check if this was a system-initiated change.
     */
    public function isSystemChange(): bool
    {
        return $this->changed_by === 'system';
    }

    /**
     * Check if this was a user-initiated change.
     */
    public function isUserChange(): bool
    {
        return !$this->isSystemChange();
    }
}
