<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;

class Payment extends Model
{
    use HasFactory;

    /**
     * The attributes that are mass assignable.
     *
     * @var array<int, string>
     */
    protected $fillable = [
        'order_id',
        'payment_method',
        'amount',
        'currency',
        'status',
        'transaction_id',
        'payment_date',
        'metadata',
    ];

    /**
     * The attributes that should be cast.
     *
     * @var array<string, string>
     */
    protected $casts = [
        'amount' => 'decimal:2',
        'payment_date' => 'datetime',
        'metadata' => 'array',
    ];

    /**
     * Payment status constants
     */
    const STATUS_PENDING = 'pending';
    const STATUS_COMPLETED = 'completed';
    const STATUS_FAILED = 'failed';
    const STATUS_REFUNDED = 'refunded';
    const STATUS_PARTIALLY_REFUNDED = 'partially_refunded'; // New constant for partial refunds

    /**
     * Payment method constants
     */
    const METHOD_PAYPAL = 'paypal';
    const METHOD_STRIPE = 'stripe';

    /**
     * Get the order that owns the payment.
     */
    public function order()
    {
        return $this->belongsTo(Order::class);
    }

    /**
     * Check if the payment is completed.
     *
     * @return bool
     */
    public function isCompleted(): bool
    {
        return $this->status === self::STATUS_COMPLETED;
    }

    /**
     * Check if the payment is pending.
     *
     * @return bool
     */
    public function isPending(): bool
    {
        return $this->status === self::STATUS_PENDING;
    }

    /**
     * Check if the payment has failed.
     *
     * @return bool
     */
    public function hasFailed(): bool
    {
        return $this->status === self::STATUS_FAILED;
    }

    /**
     * Check if the payment was refunded.
     *
     * @return bool
     */
    public function wasRefunded(): bool
    {
        return $this->status === self::STATUS_REFUNDED;
    }

    /**
     * Check if the payment was partially refunded.
     *
     * @return bool
     */
    public function isPartiallyRefunded(): bool
    {
        return $this->status === self::STATUS_PARTIALLY_REFUNDED;
    }

    /**
     * Update the payment status.
     *
     * @param string $status
     * @return bool
     */
    public function updateStatus(string $status): bool
    {
        $this->status = $status;
        return $this->save();
    }

    /**
     * Get the refunds for this payment.
     */
    public function refunds()
    {
        return $this->hasMany(Refund::class);
    }

    /**
     * Get the amount that can be refunded.
     *
     * @return float
     */
    public function getRefundableAmount(): float
    {
        // Only completed or partially refunded payments can be refunded
        if (!in_array($this->status, [self::STATUS_COMPLETED, self::STATUS_PARTIALLY_REFUNDED])) {
            return 0;
        }

        // If the payment is already marked as fully refunded
        if ($this->status === self::STATUS_REFUNDED) {
            return 0;
        }

        // Calculate already refunded amount
        $refundedAmount = $this->refunds()
            ->where('status', Refund::STATUS_COMPLETED)
            ->sum('amount');

        // The refundable amount is the original amount minus already refunded amount
        // Ensure we don't return negative values
        $refundableAmount = (float)$this->amount - (float)$refundedAmount;

        return max(0, $refundableAmount);
    }

    /**
     * Check if the payment can be refunded.
     *
     * @return bool
     */
    public function canBeRefunded(): bool
    {
        return (in_array($this->status, [self::STATUS_COMPLETED, self::STATUS_PARTIALLY_REFUNDED]))
            && $this->getRefundableAmount() > 0;
    }

    /**
     * Check if the payment is fully refunded.
     *
     * @return bool
     */
    public function isFullyRefunded(): bool
    {
        if ($this->status === self::STATUS_REFUNDED) {
            return true;
        }

        return $this->getRefundableAmount() <= 0;
    }

    /**
     * Get the total refunded amount.
     *
     * @return float
     */
    public function getRefundedAmount(): float
    {
        return (float)$this->refunds()
            ->where('status', Refund::STATUS_COMPLETED)
            ->sum('amount');
    }

    /**
     * Update payment status after a refund is processed.
     *
     * @param float $refundAmount The amount that was refunded
     * @return bool
     */
    public function updateStatusAfterRefund(float $refundAmount): bool
    {
        // Calculate the total refunded amount including this new refund
        $totalRefunded = $this->getRefundedAmount();

        // If all money is refunded (or within a small epsilon for floating point comparison)
        if (abs($totalRefunded - (float)$this->amount) < 0.01) {
            return $this->updateStatus(self::STATUS_REFUNDED);
        }
        // If some money is refunded but not all
        elseif ($totalRefunded > 0) {
            return $this->updateStatus(self::STATUS_PARTIALLY_REFUNDED);
        }

        return true;
    }
}
