<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Support\Facades\Crypt;

class PaymentMethod extends Model
{
    use HasFactory;

    /**
     * The attributes that are mass assignable.
     *
     * @var array<int, string>
     */
    protected $fillable = [
        'name',
        'display_name',
        'is_active',
        'config_data',
    ];

    /**
     * The attributes that should be cast.
     *
     * @var array<string, string>
     */
    protected $casts = [
        'is_active' => 'boolean',
        'config_data' => 'array',
    ];

    /**
     * Payment method constants
     */
    const PAYPAL = 'paypal';
    const STRIPE = 'stripe';
    const PAYPAL_DIRECT = 'paypal_direct';

    /**
     * Get all active payment methods.
     *
     * @return \Illuminate\Database\Eloquent\Collection
     */
    public static function getActive()
    {
        return self::where('is_active', true)->get();
    }

    /**
     * Check if this payment method is active.
     *
     * @return bool
     */
    public function isActive(): bool
    {
        return $this->is_active;
    }

    /**
     * Activate this payment method.
     *
     * @return bool
     */
    public function activate(): bool
    {
        $this->is_active = true;
        return $this->save();
    }

    /**
     * Deactivate this payment method.
     *
     * @return bool
     */
    public function deactivate(): bool
    {
        $this->is_active = false;
        return $this->save();
    }

    /**
     * Get a specific configuration value.
     *
     * @param string $key
     * @param mixed $default
     * @return mixed
     */
    public function getConfig(string $key, $default = null)
    {
        return $this->config_data[$key] ?? $default;
    }

    /**
     * Set a specific configuration value.
     *
     * @param string $key
     * @param mixed $value
     * @return $this
     */
    public function setConfig(string $key, $value)
    {
        $config = $this->config_data ?? [];
        $config[$key] = $value;
        $this->config_data = $config;
        return $this;
    }

    /**
     * Get the PayPal payment method.
     *
     * @return self|null
     */
    public static function getPayPal()
    {
        return self::where('name', self::PAYPAL)->first();
    }

    /**
     * Get the Stripe payment method.
     *
     * @return self|null
     */
    public static function getStripe()
    {
        return self::where('name', self::STRIPE)->first();
    }

    public static function getPayPalDirect()
    {
        return self::where('name', self::PAYPAL_DIRECT)->first();
    }
}
