<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Support\Facades\Log;

class Refund extends Model
{
    use HasFactory;

    /**
     * The attributes that are mass assignable.
     *
     * @var array<int, string>
     */
    protected $fillable = [
        'payment_id',
        'amount',
        'currency',
        'status',
        'refund_id',
        'reason',
        'notes',
        'metadata',
        'processed_at',
        'processed_by',
    ];

    /**
     * The attributes that should be cast.
     *
     * @var array<string, string>
     */
    protected $casts = [
        'amount' => 'decimal:2',
        'metadata' => 'array',
        'processed_at' => 'datetime',
    ];

    /**
     * Refund status constants
     */
    const STATUS_PENDING = 'pending';
    const STATUS_COMPLETED = 'completed';
    const STATUS_FAILED = 'failed';

    /**
     * Get the payment that was refunded.
     */
    public function payment()
    {
        return $this->belongsTo(Payment::class);
    }

    /**
     * Get the order associated with this refund (through payment).
     */
    public function order()
    {
        return $this->payment->order();
    }

    /**
     * Get the user who processed the refund.
     */
    public function processor()
    {
        return $this->belongsTo(User::class, 'processed_by');
    }

    /**
     * Check if the refund is completed.
     *
     * @return bool
     */
    public function isCompleted(): bool
    {
        return $this->status === self::STATUS_COMPLETED;
    }

    /**
     * Check if the refund is pending.
     *
     * @return bool
     */
    public function isPending(): bool
    {
        return $this->status === self::STATUS_PENDING;
    }

    /**
     * Check if the refund has failed.
     *
     * @return bool
     */
    public function hasFailed(): bool
    {
        return $this->status === self::STATUS_FAILED;
    }

    /**
     * Mark the refund as completed and update related payment and order statuses.
     *
     * @param string|null $refundId
     * @param array $metadata
     * @param bool $updateRelated Whether to update related payment and order statuses
     * @return bool
     */
    public function markAsCompleted(?string $refundId = null, array $metadata = [], bool $updateRelated = true): bool
    {
        $this->status = self::STATUS_COMPLETED;
        $this->processed_at = now();

        if ($refundId) {
            $this->refund_id = $refundId;
        }

        if (!empty($metadata)) {
            $this->metadata = array_merge($this->metadata ?? [], $metadata);
        }

        $saved = $this->save();

        // Update related payment and order statuses if requested
        if ($saved && $updateRelated) {
            try {
                // Update payment status
                if ($this->payment) {
                    $this->payment->updateStatusAfterRefund((float)$this->amount);

                    // Update order status
                    if ($this->payment->order) {
                        $this->payment->order->updatePaymentStatusAfterRefund(
                            $this->payment,
                            (float)$this->amount
                        );
                    }
                }
            } catch (\Exception $e) {
                Log::error('Error updating statuses after refund completion', [
                    'refund_id' => $this->id,
                    'error' => $e->getMessage(),
                    'trace' => $e->getTraceAsString()
                ]);
            }
        }

        return $saved;
    }

    /**
     * Mark the refund as failed.
     *
     * @param string $reason
     * @param array $metadata
     * @return bool
     */
    public function markAsFailed(string $reason, array $metadata = []): bool
    {
        $this->status = self::STATUS_FAILED;
        $this->notes = $reason;

        if (!empty($metadata)) {
            $this->metadata = array_merge($this->metadata ?? [], $metadata);
        }

        return $this->save();
    }

    /**
     * Calculate the percentage of the original payment that this refund represents.
     *
     * @return float Percentage value (0-100)
     */
    public function getRefundPercentage(): float
    {
        if (!$this->payment || (float)$this->payment->amount <= 0) {
            return 0;
        }

        return ((float)$this->amount / (float)$this->payment->amount) * 100;
    }

    /**
     * Determine if this refund is for the full payment amount.
     *
     * @return bool
     */
    public function isFullRefund(): bool
    {
        if (!$this->payment) {
            return false;
        }

        // Use a small epsilon for floating point comparison
        return abs((float)$this->amount - (float)$this->payment->amount) < 0.01;
    }

    /**
     * Get all refunds for a specific order.
     *
     * @param int $orderId
     * @return \Illuminate\Database\Eloquent\Collection
     */
    public static function getRefundsForOrder(int $orderId)
    {
        return self::whereHas('payment', function ($query) use ($orderId) {
            $query->where('order_id', $orderId);
        })->get();
    }

    /**
     * Get the total refunded amount for a specific order.
     *
     * @param int $orderId
     * @return float
     */
    public static function getTotalRefundedForOrder(int $orderId): float
    {
        return (float)self::whereHas('payment', function ($query) use ($orderId) {
            $query->where('order_id', $orderId);
        })->where('status', self::STATUS_COMPLETED)
            ->sum('amount');
    }
}
