<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Relations\BelongsTo;

class UserCoupon extends Model
{
    use HasFactory;

    /**
     * The attributes that are mass assignable.
     *
     * @var array<int, string>
     */
    protected $fillable = [
        'user_id',
        'coupon_id',
        'used_at',
        'order_total',
        'discount_amount',
        'order_id',
    ];

    /**
     * The attributes that should be cast.
     *
     * @var array<string, string>
     */
    protected $casts = [
        'used_at' => 'datetime',
        'order_total' => 'decimal:2',
        'discount_amount' => 'decimal:2',
    ];

    /**
     * Get the user who used this coupon.
     */
    public function user(): BelongsTo
    {
        return $this->belongsTo(User::class);
    }

    /**
     * Get the coupon that was used.
     */
    public function coupon(): BelongsTo
    {
        return $this->belongsTo(Coupon::class);
    }

    /**
     * Get the order where this coupon was applied.
     */
    public function order(): BelongsTo
    {
        return $this->belongsTo(Order::class);
    }

    /**
     * Get usage count for a user and coupon combination.
     *
     * @param int $userId
     * @param int $couponId
     * @return int
     */
    public static function getUserUsageCount(int $userId, int $couponId): int
    {
        return self::where('user_id', $userId)
            ->where('coupon_id', $couponId)
            ->count();
    }

    /**
     * Get all coupons used by a specific user.
     *
     * @param int $userId
     * @return \Illuminate\Database\Eloquent\Collection
     */
    public static function getCouponsUsedByUser(int $userId)
    {
        return self::where('user_id', $userId)
            ->with('coupon')
            ->orderBy('used_at', 'desc')
            ->get();
    }

    /**
     * Record the usage of a coupon by a user.
     *
     * @param int $userId
     * @param int $couponId
     * @param float $orderTotal
     * @param float $discountAmount
     * @param int|null $orderId
     * @return self
     */
    public static function recordUsage(
        int $userId,
        int $couponId,
        float $orderTotal,
        float $discountAmount,
        ?int $orderId = null
    ): self {
        return self::create([
            'user_id' => $userId,
            'coupon_id' => $couponId,
            'used_at' => now(),
            'order_total' => $orderTotal,
            'discount_amount' => $discountAmount,
            'order_id' => $orderId,
        ]);
    }
}
