<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;

class WriterEssayTest extends Model
{
    use HasFactory;

    /**
     * The attributes that are mass assignable.
     *
     * @var array<int, string>
     */
    protected $fillable = [
        'writer_profile_id',
        'essay_test_status',
        'essay_started_at',
        'essay_end_at',
        'essay_completed_at',
        'essay_score',
        'essay_feedback',
        'content', // Legacy field for directly written essays
        // File upload fields
        'file_path',
        'original_filename',
        'file_type',
        'file_size',
        'upload_time',
        'is_file_validated',
        'validation_message',
        // Retry control fields
        'last_failed_at',
        'can_retry_after',
        // Security fields
        'started_at',              // When the test was started (server time)
        'expected_end_at',         // When the test should end based on duration
        'last_activity_at',        // Last recorded activity during the test
        'time_extensions',         // Number of legitimate time extensions granted
        'session_token',           // Unique token for this test session
        'ip_address',              // IP address used during the test
        'user_agent',              // Browser/device information
        'is_time_violation',       // Flag for suspicious timing behavior
        'time_violation_notes',    // Notes about any timing violations
        'auto_processed_at',       // When the test was automatically processed
    ];

    /**
     * The attributes that should be cast.
     *
     * @var array<string, string>
     */
    protected $casts = [
        'essay_started_at' => 'datetime',
        'essay_end_at' => 'datetime',
        'essay_completed_at' => 'datetime',
        'essay_score' => 'integer',
        'upload_time' => 'datetime',
        'is_file_validated' => 'boolean',
        'last_failed_at' => 'datetime',
        'can_retry_after' => 'datetime',
        // Security field casts
        'started_at' => 'datetime',
        'expected_end_at' => 'datetime',
        'last_activity_at' => 'datetime',
        'auto_processed_at' => 'datetime',
        'time_extensions' => 'integer',
        'is_time_violation' => 'boolean',
    ];

    /**
     * Get the writer profile that owns the essay test.
     */
    public function writerProfile()
    {
        return $this->belongsTo(WriterProfile::class);
    }

    /**
     * Check if the writer can take the essay test.
     */
    public function canTakeEssay()
    {
        // Writer can take the test if they haven't started
        if ($this->essay_test_status === 'not_started') {
            return true;
        }

        // Check if they failed or expired and waiting period is over
        if (($this->essay_test_status === 'essay_failed' || $this->essay_test_status === 'essay_expired')
            && $this->can_retry_after
        ) {
            return now()->gte($this->can_retry_after);
        }

        return false;
    }


    /**
     * Start a new test session
     *
     * @param int $durationMinutes Test duration in minutes
     * @param string $ipAddress User's IP address
     * @param string $userAgent User's browser/device information
     * @return string Session token
     */
    public function startTestSession($durationMinutes, $ipAddress, $userAgent)
    {
        $now = now();
        // Create a separate variable for the end time by cloning $now
        $endTime = $now->copy()->addMinutes($durationMinutes);

        $sessionToken = md5($this->writer_profile_id . $now->timestamp . rand(1000, 9999));

        $this->update([
            'essay_test_status' => 'essay_pending',
            'essay_started_at' => $now,
            'essay_end_at' => $endTime,
            'started_at' => $now,
            'expected_end_at' => $endTime,
            'last_activity_at' => $now,
            'session_token' => $sessionToken,
            'ip_address' => $ipAddress,
            'user_agent' => $userAgent,
            'is_time_violation' => false,
            'time_violation_notes' => null,
            'time_extensions' => 0,
            // Reset file upload fields if retaking test
            'file_path' => '',
            'original_filename' => '',
            'file_type' => '',
            'file_size' => 0,
            'upload_time' => null,
            'is_file_validated' => false,
            'validation_message' => null,
            'auto_processed_at' => null
        ]);

        return $sessionToken;
    }

    /**
     * Check if the test is still within the allowed time window
     *
     * @return bool
     */
    public function isWithinTimeWindow()
    {
        if (!$this->expected_end_at) {
            return false;
        }

        return now()->lte($this->expected_end_at);
    }

    /**
     * Get remaining test time in seconds
     *
     * @return int Seconds remaining (0 if expired)
     */
    public function getRemainingTime()
    {
        if (!$this->expected_end_at) {
            return 0;
        }

        $remainingSeconds = now()->diffInSeconds($this->expected_end_at, false);
        return max(0, $remainingSeconds);
    }

    /**
     * Record user activity during the test
     *
     * @param string $sessionToken Current session token
     * @param string $ipAddress Current IP address
     * @return bool Whether the activity was recorded successfully
     */
    public function recordActivity($sessionToken, $ipAddress)
    {
        // Verify session token matches
        if ($this->session_token !== $sessionToken) {
            $this->flagTimeViolation('Session token mismatch');
            return false;
        }

        // Check for IP address changes
        if ($this->ip_address !== $ipAddress && $this->ip_address !== null) {
            $this->flagTimeViolation('IP address changed during test');
        }

        $this->update([
            'last_activity_at' => now()
        ]);

        return true;
    }

    /**
     * Flag a time violation
     *
     * @param string $notes Notes about the violation
     */
    public function flagTimeViolation($notes)
    {
        $this->update([
            'is_time_violation' => true,
            'time_violation_notes' => $notes
        ]);
    }

    /**
     * Verify test submission timing
     *
     * @param string $sessionToken Session token for verification
     * @return bool Whether the submission timing is valid
     */
    public function verifySubmissionTiming($sessionToken)
    {
        // Verify session token
        if ($this->session_token !== $sessionToken) {
            $this->flagTimeViolation('Invalid session token on submission');
            return false;
        }

        // Check if test is still within time window
        if (!$this->isWithinTimeWindow()) {
            $this->flagTimeViolation('Submission after time expired');
            return false;
        }

        // Check for suspicious gaps in activity
        if ($this->last_activity_at && now()->diffInMinutes($this->last_activity_at) > 5) {
            $this->flagTimeViolation('Suspicious gap in activity before submission');
            // We still allow the submission but flag it
        }

        return true;
    }

    /**
     * Store uploaded file information
     *
     * @param string $filePath Path where the file is stored
     * @param string $originalFilename Original name of the uploaded file
     * @param string $fileType Type of the file (docx, pdf)
     * @param int $fileSize Size of the file in bytes
     * @return bool Whether the file information was stored successfully
     */
    public function storeFileInformation($filePath, $originalFilename, $fileType, $fileSize)
    {
        return $this->update([
            'file_path' => $filePath,
            'original_filename' => $originalFilename,
            'file_type' => $fileType,
            'file_size' => $fileSize,
            'upload_time' => now(),
            'is_file_validated' => true,
            'validation_message' => 'File successfully uploaded and validated'
        ]);
    }

    /**
     * Mark file validation failure
     *
     * @param string $message Error message explaining the validation failure
     * @return bool Whether the validation failure was recorded successfully
     */
    public function markFileValidationFailure($message)
    {
        return $this->update([
            'is_file_validated' => false,
            'validation_message' => $message
        ]);
    }

    /**
     * Check if a file has been uploaded for this test
     *
     * @return bool Whether a file has been uploaded
     */
    public function hasUploadedFile()
    {
        return !empty($this->file_path) && $this->is_file_validated;
    }

    /**
     * Complete the essay test with the uploaded file
     *
     * @param string $sessionToken Session token for verification
     * @return bool Whether the test was completed successfully
     */
    public function completeTest($sessionToken)
    {
        // Verify submission timing
        if (!$this->verifySubmissionTiming($sessionToken)) {
            return false;
        }

        // Check if a file has been uploaded
        if (!$this->hasUploadedFile()) {
            $this->markFileValidationFailure('No valid file uploaded for submission');
            return false;
        }

        // Mark test as completed
        $this->update([
            'essay_test_status' => 'essay_submitted',
            'essay_completed_at' => now()
        ]);

        return true;
    }

    /**
     * Check if the test has expired without submission
     *
     * @return bool
     */
    public function hasExpiredWithoutSubmission()
    {
        return $this->essay_test_status === 'essay_pending' &&
            $this->expected_end_at &&
            now()->gt($this->expected_end_at);
    }

    /**
     * Mark test as expired
     *
     * @param string|null $notes Additional notes about expiration
     * @return bool
     */
    public function markAsExpired($notes = null)
    {
        // Set retry date to 3 months from now if no file was uploaded
        $retryDate = now()->addMonths(3);

        return $this->update([
            'essay_test_status' => 'essay_expired',
            'essay_completed_at' => now(),
            'time_violation_notes' => $notes ?? 'Test expired without submission',
            'auto_processed_at' => now(),
            'last_failed_at' => now(),
            'can_retry_after' => $retryDate
        ]);
    }


    /**
     * Process partial submission (file uploaded but not submitted)
     *
     * @return bool
     */
    public function processPartialSubmission()
    {
        // Only process if there's an uploaded file
        if (!$this->hasUploadedFile()) {
            return $this->markAsExpired('Test expired with no file uploaded');
        }

        // Mark as submitted but flag it as auto-processed
        return $this->update([
            'essay_test_status' => 'essay_submitted',
            'essay_completed_at' => now(),
            'time_violation_notes' => 'Partial submission auto-processed after deadline',
            'auto_processed_at' => now()
        ]);
    }

    /**
     * Auto-submit the test based on client-side expiration
     *
     * @param string $sessionToken Session token for verification
     * @return bool Whether the auto-submission was successful
     */
    public function autoSubmit($sessionToken)
    {
        // Verify session token
        if ($this->session_token !== $sessionToken) {
            $this->flagTimeViolation('Invalid session token on auto-submission');
            return false;
        }

        // Check if the test has actually expired
        if (!$this->hasExpiredWithoutSubmission()) {
            return false;
        }

        // Process based on whether a file was uploaded
        if ($this->hasUploadedFile()) {
            return $this->update([
                'essay_test_status' => 'essay_submitted',
                'essay_completed_at' => now(),
                'time_violation_notes' => 'Auto-submitted by client when time expired',
                'auto_processed_at' => now()
            ]);
        } else {
            return $this->markAsExpired('Auto-processed by client - no file uploaded');
        }
    }
}
