<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Relations\BelongsTo;

class WriterFine extends Model
{
    use HasFactory;

    protected $fillable = [
        'writer_id',
        'infraction_id',
        'order_id',
        'applied_by',
        'reviewed_by',
        'fine_type',
        'fine_amount',
        'currency',
        'fine_reason',
        'calculation_details',
        'application_method',
        'status',
        'applied_at',
        'effective_from',
        'deducted_at',
        'resolved_at',
        'amount_deducted',
        'remaining_amount',
        'payment_record_id',
        'writer_appeal',
        'appeal_decision',
        'review_notes',
        'fine_configuration',
        'is_automatic',
        'manual_adjustment_notes',
        'related_fine_id',
        'installment_number',
    ];

    protected $casts = [
        'applied_at' => 'datetime',
        'effective_from' => 'datetime',
        'deducted_at' => 'datetime',
        'resolved_at' => 'datetime',
        'fine_amount' => 'decimal:2',
        'amount_deducted' => 'decimal:2',
        'remaining_amount' => 'decimal:2',
        'fine_configuration' => 'array',
        'is_automatic' => 'boolean',
    ];

    /**
     * Get the writer who received the fine.
     */
    public function writer(): BelongsTo
    {
        return $this->belongsTo(User::class, 'writer_id');
    }

    /**
     * Get the infraction that triggered this fine.
     */
    public function infraction(): BelongsTo
    {
        return $this->belongsTo(WriterInfraction::class, 'infraction_id');
    }

    /**
     * Get the order related to this fine.
     */
    public function order(): BelongsTo
    {
        return $this->belongsTo(Order::class);
    }

    /**
     * Get the admin who applied the fine.
     */
    public function appliedBy(): BelongsTo
    {
        return $this->belongsTo(User::class, 'applied_by');
    }

    /**
     * Get the admin who reviewed the fine.
     */
    public function reviewedBy(): BelongsTo
    {
        return $this->belongsTo(User::class, 'reviewed_by');
    }

    /**
     * Get the payment record where the fine was deducted.
     */
    public function paymentRecord(): BelongsTo
    {
        return $this->belongsTo(WriterPaymentRecord::class, 'payment_record_id');
    }

    /**
     * Get the related fine if this is part of a series.
     */
    public function relatedFine(): BelongsTo
    {
        return $this->belongsTo(WriterFine::class, 'related_fine_id');
    }

    /**
     * Check if the fine is effective (can be deducted).
     */
    public function isEffective(): bool
    {
        if (!$this->effective_from) {
            return true; // No effective date means immediately effective
        }
        
        return now()->isAfter($this->effective_from);
    }

    /**
     * Check if the fine has been fully deducted.
     */
    public function isFullyDeducted(): bool
    {
        return $this->remaining_amount <= 0;
    }

    /**
     * Check if the fine can be appealed.
     */
    public function canBeAppealed(): bool
    {
        return in_array($this->status, ['pending', 'applied']) && 
               !$this->isFullyDeducted();
    }

    /**
     * Calculate the remaining amount to be deducted.
     */
    public function getRemainingAmount(): float
    {
        return max(0, $this->fine_amount - $this->amount_deducted);
    }

    /**
     * Check if this is an installment payment.
     */
    public function isInstallment(): bool
    {
        return !is_null($this->installment_number);
    }

    /**
     * Get the fine amount for this installment.
     */
    public function getInstallmentAmount(): float
    {
        if (!$this->isInstallment()) {
            return $this->fine_amount;
        }
        
        // For now, assume equal installments
        // This could be enhanced with custom installment logic
        $totalInstallments = $this->relatedFine ? 
            WriterFine::where('related_fine_id', $this->related_fine_id)->count() : 1;
        
        return $this->fine_amount / $totalInstallments;
    }

    /**
     * Apply the fine to a payment record.
     */
    public function applyToPayment(WriterPaymentRecord $paymentRecord): bool
    {
        if (!$this->isEffective() || $this->isFullyDeducted()) {
            return false;
        }
        
        $deductibleAmount = min($this->remaining_amount, $paymentRecord->payment_amount);
        
        if ($deductibleAmount > 0) {
            $this->amount_deducted += $deductibleAmount;
            $this->remaining_amount = $this->getRemainingAmount();
            $this->payment_record_id = $paymentRecord->id;
            $this->deducted_at = now();
            
            if ($this->isFullyDeducted()) {
                $this->status = 'applied';
            }
            
            $this->save();
            
            return true;
        }
        
        return false;
    }

    /**
     * Scope for active fines.
     */
    public function scopeActive($query)
    {
        return $query->whereIn('status', ['pending', 'applied']);
    }

    /**
     * Scope for pending fines.
     */
    public function scopePending($query)
    {
        return $query->where('status', 'pending');
    }

    /**
     * Scope for applied fines.
     */
    public function scopeApplied($query)
    {
        return $query->where('status', 'applied');
    }

    /**
     * Scope for automatic fines.
     */
    public function scopeAutomatic($query)
    {
        return $query->where('is_automatic', true);
    }

    /**
     * Scope for manual fines.
     */
    public function scopeManual($query)
    {
        return $query->where('is_automatic', false);
    }

    /**
     * Scope for fines by type.
     */
    public function scopeByType($query, $type)
    {
        return $query->where('fine_type', $type);
    }

    /**
     * Scope for fines by application method.
     */
    public function scopeByApplicationMethod($query, $method)
    {
        return $query->where('application_method', $method);
    }

    /**
     * Scope for fines that need review.
     */
    public function scopeNeedsReview($query)
    {
        return $query->whereIn('status', ['pending', 'disputed']);
    }
} 