<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Relations\BelongsTo;
use Illuminate\Database\Eloquent\Relations\HasMany;

class WriterInfraction extends Model
{
    use HasFactory;

    protected $fillable = [
        'writer_id',
        'order_id',
        'reported_by',
        'reviewed_by',
        'infraction_type',
        'severity_level',
        'description',
        'evidence',
        'admin_notes',
        'status',
        'occurred_at',
        'reported_at',
        'reviewed_at',
        'resolved_at',
        'appeal_deadline',
        'writer_response',
        'appeal_reason',
        'appeal_decision_notes',
        'estimated_cost_impact',
        'client_satisfaction_impact',
        'impact_assessment',
        'repeat_count',
        'escalated',
        'escalation_notes',
        'related_infraction_id',
    ];

    protected $casts = [
        'occurred_at' => 'datetime',
        'reported_at' => 'datetime',
        'reviewed_at' => 'datetime',
        'resolved_at' => 'datetime',
        'appeal_deadline' => 'datetime',
        'escalated' => 'boolean',
        'estimated_cost_impact' => 'decimal:2',
        'client_satisfaction_impact' => 'integer',
        'repeat_count' => 'integer',
    ];

    /**
     * Get the writer who committed the infraction.
     */
    public function writer(): BelongsTo
    {
        return $this->belongsTo(User::class, 'writer_id');
    }

    /**
     * Get the order related to this infraction.
     */
    public function order(): BelongsTo
    {
        return $this->belongsTo(Order::class);
    }

    /**
     * Get the admin who reported the infraction.
     */
    public function reportedBy(): BelongsTo
    {
        return $this->belongsTo(User::class, 'reported_by');
    }

    /**
     * Get the admin who reviewed the infraction.
     */
    public function reviewedBy(): BelongsTo
    {
        return $this->belongsTo(User::class, 'reviewed_by');
    }

    /**
     * Get the related infraction if this is a repeat offense.
     */
    public function relatedInfraction(): BelongsTo
    {
        return $this->belongsTo(WriterInfraction::class, 'related_infraction_id');
    }

    /**
     * Get all related infractions (including this one).
     */
    public function allRelatedInfractions(): HasMany
    {
        return $this->hasMany(WriterInfraction::class, 'related_infraction_id');
    }

    /**
     * Get all fines associated with this infraction.
     */
    public function fines(): HasMany
    {
        return $this->hasMany(WriterFine::class, 'infraction_id');
    }

    /**
     * Check if the infraction can be appealed.
     */
    public function canBeAppealed(): bool
    {
        if ($this->status !== 'confirmed') {
            return false;
        }
        
        if (!$this->appeal_deadline) {
            return false;
        }
        
        return now()->isBefore($this->appeal_deadline);
    }

    /**
     * Check if the appeal deadline has passed.
     */
    public function isAppealDeadlinePassed(): bool
    {
        if (!$this->appeal_deadline) {
            return false;
        }
        
        return now()->isAfter($this->appeal_deadline);
    }

    /**
     * Check if this is a repeat offense.
     */
    public function isRepeatOffense(): bool
    {
        return $this->repeat_count > 1;
    }

    /**
     * Get the total financial impact of this infraction.
     */
    public function getTotalFinancialImpact(): float
    {
        $impact = $this->estimated_cost_impact;
        
        // Add fines
        $fines = $this->fines()->where('status', 'applied')->sum('fine_amount');
        $impact += $fines;
        
        return $impact;
    }

    /**
     * Check if this infraction should trigger escalation.
     */
    public function shouldTriggerEscalation(): bool
    {
        // Critical infractions always trigger escalation
        if ($this->severity_level === 'critical') {
            return true;
        }
        
        // Major infractions trigger escalation after 2 occurrences
        if ($this->severity_level === 'major' && $this->repeat_count >= 2) {
            return true;
        }
        
        // Minor infractions trigger escalation after 3 occurrences
        if ($this->severity_level === 'minor' && $this->repeat_count >= 3) {
            return true;
        }
        
        return false;
    }

    /**
     * Scope for active infractions.
     */
    public function scopeActive($query)
    {
        return $query->whereIn('status', ['reported', 'under_review', 'confirmed']);
    }

    /**
     * Scope for resolved infractions.
     */
    public function scopeResolved($query)
    {
        return $query->whereIn('status', ['resolved', 'appeal_approved', 'appeal_rejected']);
    }

    /**
     * Scope for infractions by severity.
     */
    public function scopeBySeverity($query, $severity)
    {
        return $query->where('severity_level', $severity);
    }

    /**
     * Scope for infractions by type.
     */
    public function scopeByType($query, $type)
    {
        return $query->where('infraction_type', $type);
    }

    /**
     * Scope for repeat offenses.
     */
    public function scopeRepeatOffenses($query)
    {
        return $query->where('repeat_count', '>', 1);
    }

    /**
     * Scope for escalated infractions.
     */
    public function scopeEscalated($query)
    {
        return $query->where('escalated', true);
    }

    /**
     * Scope for infractions that need review.
     */
    public function scopeNeedsReview($query)
    {
        return $query->whereIn('status', ['reported', 'under_review']);
    }
} 