<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Relations\BelongsTo;

class WriterLevelHistory extends Model
{
    use HasFactory;

    protected $table = 'writer_level_history';

    protected $fillable = [
        'writer_id',
        'user_id',
        'writer_level_id',
        'previous_writer_level_id',
        'change_type',
        'change_reason',
        'performance_metrics_at_change',
        'changed_by',
        'changed_by_user_id',
        'changed_at',
    ];

    protected $casts = [
        'performance_metrics_at_change' => 'array',
        'changed_at' => 'datetime',
    ];

    /**
     * Get the writer user.
     */
    public function writer(): BelongsTo
    {
        return $this->belongsTo(User::class, 'writer_id');
    }

    /**
     * Get the user profile.
     */
    public function user(): BelongsTo
    {
        return $this->belongsTo(User::class, 'user_id');
    }

    /**
     * Get the current writer level.
     */
    public function writerLevel(): BelongsTo
    {
        return $this->belongsTo(WriterLevel::class);
    }

    /**
     * Get the previous writer level.
     */
    public function previousWriterLevel(): BelongsTo
    {
        return $this->belongsTo(WriterLevel::class, 'previous_writer_level_id');
    }

    /**
     * Get the user who made the change.
     */
    public function changedByUser(): BelongsTo
    {
        return $this->belongsTo(User::class, 'changed_by_user_id');
    }

    /**
     * Check if this was a promotion.
     */
    public function isPromotion(): bool
    {
        return $this->change_type === 'promotion';
    }

    /**
     * Check if this was a demotion.
     */
    public function isDemotion(): bool
    {
        return $this->change_type === 'demotion';
    }

    /**
     * Check if this was an initial assignment.
     */
    public function isInitialAssignment(): bool
    {
        return $this->change_type === 'initial_assignment';
    }

    /**
     * Check if this was a system change.
     */
    public function isSystemChange(): bool
    {
        return $this->changed_by === 'system';
    }

    /**
     * Check if this was an admin change.
     */
    public function isAdminChange(): bool
    {
        return $this->changed_by === 'admin';
    }

    /**
     * Get the change description for display.
     */
    public function getChangeDescription(): string
    {
        $currentLevel = $this->writerLevel?->name ?? 'Unknown';
        $previousLevel = $this->previousWriterLevel?->name ?? 'None';
        
        switch ($this->change_type) {
            case 'promotion':
                return "Promoted from {$previousLevel} to {$currentLevel}";
            case 'demotion':
                return "Demoted from {$previousLevel} to {$currentLevel}";
            case 'initial_assignment':
                return "Assigned initial level: {$currentLevel}";
            default:
                return "Level changed to {$currentLevel}";
        }
    }
} 