<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Relations\BelongsTo;

class WriterPerformanceMetric extends Model
{
    use HasFactory;

    protected $fillable = [
        'writer_id',
        'user_id',
        'total_assigned_orders',
        'total_completed_orders',
        'completion_rate',
        'total_on_time_deliveries',
        'total_late_deliveries',
        'on_time_delivery_rate',
        'total_bids_placed',
        'total_accepted_bids',
        'bid_acceptance_rate',
        'total_revision_requests',
        'average_revision_rate',
        'total_pages_written',
        'average_pages_per_order',
        'academic_level_performance',
        'paper_type_performance',
        'first_order_date',
        'last_order_date',
        'days_on_platform',
        'consecutive_active_months',
        'overall_performance_score',
        'last_calculated_at',
    ];

    protected $casts = [
        'academic_level_performance' => 'array',
        'paper_type_performance' => 'array',
        'first_order_date' => 'datetime',
        'last_order_date' => 'datetime',
        'last_calculated_at' => 'datetime',
        'completion_rate' => 'decimal:2',
        'on_time_delivery_rate' => 'decimal:2',
        'bid_acceptance_rate' => 'decimal:2',
        'average_revision_rate' => 'decimal:2',
        'average_pages_per_order' => 'decimal:2',
        'overall_performance_score' => 'decimal:2',
    ];

    /**
     * Get the writer user.
     */
    public function writer(): BelongsTo
    {
        return $this->belongsTo(User::class, 'writer_id');
    }

    /**
     * Get the user profile.
     */
    public function user(): BelongsTo
    {
        return $this->belongsTo(User::class, 'user_id');
    }

    /**
     * Get the writer profile.
     */
    public function writerProfile(): BelongsTo
    {
        return $this->belongsTo(WriterProfile::class, 'writer_id', 'user_id');
    }

    /**
     * Check if writer meets criteria for a specific level.
     */
    public function meetsLevelCriteria(int $levelRank): bool
    {
        switch ($levelRank) {
            case 1: // Entry Level
                return true; // Everyone starts here
                
            case 2: // Standard Writer
                return $this->total_completed_orders >= 20 &&
                       $this->completion_rate >= 90.0 &&
                       $this->on_time_delivery_rate >= 90.0;
                
            case 3: // Advanced Writer
                return $this->total_completed_orders >= 50 &&
                       $this->completion_rate >= 95.0 &&
                       $this->on_time_delivery_rate >= 95.0 &&
                       $this->days_on_platform >= 180; // 6 months
                
            case 4: // Expert Writer
                return $this->total_completed_orders >= 100 &&
                       $this->completion_rate >= 98.0 &&
                       $this->on_time_delivery_rate >= 98.0 &&
                       $this->days_on_platform >= 365; // 1 year
                
            case 5: // Premium Writer
                return $this->total_completed_orders >= 200 &&
                       $this->completion_rate >= 99.0 &&
                       $this->on_time_delivery_rate >= 99.0 &&
                       $this->days_on_platform >= 730; // 2 years
                
            case 6: // Distinguished Scholar
                return $this->total_completed_orders >= 500 &&
                       $this->completion_rate >= 99.5 &&
                       $this->on_time_delivery_rate >= 99.5 &&
                       $this->days_on_platform >= 1095; // 3 years
                
            default:
                return false;
        }
    }

    /**
     * Get the recommended writer level based on current performance.
     */
    public function getRecommendedLevel(): int
    {
        for ($level = 6; $level >= 1; $level--) {
            if ($this->meetsLevelCriteria($level)) {
                return $level;
            }
        }
        return 1; // Default to Entry Level
    }

    /**
     * Check if writer qualifies for promotion.
     */
    public function qualifiesForPromotion(int $currentLevelRank): bool
    {
        $recommendedLevel = $this->getRecommendedLevel();
        return $recommendedLevel > $currentLevelRank;
    }

    /**
     * Check if writer should be demoted.
     */
    public function shouldBeDemoted(int $currentLevelRank): bool
    {
        $recommendedLevel = $this->getRecommendedLevel();
        return $recommendedLevel < $currentLevelRank;
    }
} 