<?php

namespace App\Models;

use App\Models\WriterPaymentAccount;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Support\Facades\Storage;
use Illuminate\Database\Eloquent\Factories\HasFactory;

class WriterProfile extends Model
{
    use HasFactory;

    /**
     * The attributes that are mass assignable.
     *
     * @var array<int, string>
     */
    protected $fillable = [
        'user_id',
        'nickname',
        'profile_image',
        'bio',
        'city_id',
        'citation_style_id',
        'education_level_id',
        'is_available',
        'payment_method_id',
        'payment_details',
        'specialization_id',
        'writer_level_id',
        'old_test_status', // Legacy field
        'last_active_at',
        'last_bid_at',
        'experience_years',
        'hourly_rate',
    ];

    /**
     * The attributes that should be cast.
     *
     * @var array<string, string>
     */
    protected $casts = [
        'last_active_at' => 'datetime',
        'last_bid_at' => 'datetime',
        'is_available' => 'boolean',
        'hourly_rate' => 'decimal:2',
    ];

    /**
     * Get the user that owns the writer profile.
     */
    public function user()
    {
        return $this->belongsTo(User::class);
    }

    /**
     * Get the city associated with the writer profile.
     */
    public function city()
    {
        return $this->belongsTo(City::class);
    }

    /**
     * Get the citation styles associated with the writer profile.
     */
    public function citationStyles()
    {
        return $this->belongsToMany(WriterCitationStyle::class, 'writer_citation_style');
    }


    /**
     * Get the education level associated with the writer profile.
     */
    public function educationLevel()
    {
        return $this->belongsTo(EducationLevel::class);
    }

    /**
     * Get the payment method associated with the writer profile.
     */
    public function paymentMethod()
    {
        return $this->belongsTo(PaymentMethod::class);
    }

    /**
     * Get the specialization associated with the writer profile.
     */
    public function specialization()
    {
        return $this->belongsTo(Specialization::class);
    }

    /**
     * Get the writer level associated with the writer profile.
     */
    public function writerLevel()
    {
        return $this->belongsTo(WriterLevel::class);
    }

    /**
     * Get the subjects associated with the writer profile.
     */

    public function subjects()
    {
        return $this->belongsToMany(Subject::class, 'writer_subjects');
    }


    /**
     * Get the languages associated with the writer profile.
     */
    public function languages()
    {
        return $this->belongsToMany(Language::class, 'writer_languages');
    }


    /**
     * Get the work samples associated with the writer profile.
     */
    public function workSamples()
    {
        return $this->hasMany(WriterWorkSample::class);
    }

    /**
     * Get the qualification test associated with the writer profile.
     */
    public function qualificationTest()
    {
        return $this->hasOne(WriterQualificationTest::class);
    }

    /**
     * Get the essay test associated with the writer profile.
     */
    public function essayTest()
    {
        return $this->hasOne(WriterEssayTest::class);
    }

    /**
     * Check if the writer can take the multiple-choice test.
     */
    /**
     * Check if the writer can take the multiple-choice test.
     */
    public function canTakeTest()
    {
        // If no qualification test exists yet, they can take it
        if (!$this->qualificationTest) {
            return true;
        }

        // If a test is in progress and within time window, they should continue it
        if (
            $this->qualificationTest->questions_test_status === 'questions_pending' &&
            $this->qualificationTest->isWithinTimeWindow()
        ) {
            return false; // They should continue their current test
        }

        // Writer can take the test if they haven't started or if they failed but the waiting period is over
        if ($this->qualificationTest->questions_test_status === 'not_started') {
            return true;
        }

        if (
            $this->qualificationTest->questions_test_status === 'questions_failed' &&
            $this->qualificationTest->can_retry_after
        ) {
            return now()->gte($this->qualificationTest->can_retry_after);
        }

        return false;
    }

    /**
     * Check if the writer has a test in progress that they should continue
     */
    public function hasTestInProgress()
    {
        return $this->qualificationTest &&
            $this->qualificationTest->questions_test_status === 'questions_pending' &&
            $this->qualificationTest->isWithinTimeWindow();
    }


    /**
     * Check if the writer can take the essay test.
     */
    public function canTakeEssay()
    {
        // Writer can only take the essay test if they've passed the qualification test
        if (!$this->qualificationTest || $this->qualificationTest->questions_test_status !== 'passed') {
            return false;
        }

        return $this->essayTest ? $this->essayTest->canTakeEssay() : true;
    }

    /**
     * Check if the writer is currently taking the essay test.
     */
    public function isEssayInProgress()
    {
        return $this->essayTest ? $this->essayTest->isEssayInProgress() : false;
    }

    /**
     * Get the overall qualification status of the writer.
     *
     * Logic:
     * - If questions test is not passed, return questions status
     * - If questions test is passed but essay is not passed, return essay status
     * - Only return 'passed' if both tests are passed
     */
    public function getQualificationStatus()
    {
        // If no qualification test exists yet
        if (!$this->qualificationTest) {
            return 'not_started';
        }

        // If questions test is not passed, that's the overall status
        if ($this->qualificationTest->questions_test_status !== 'passed') {
            return $this->qualificationTest->questions_test_status;
        }

        // If no essay test exists yet
        if (!$this->essayTest) {
            return 'questions_passed_essay_not_started';
        }

        // If questions test is passed but essay is not passed, essay status is the overall status
        if ($this->essayTest->essay_test_status !== 'passed') {
            return $this->essayTest->essay_test_status;
        }

        // Both tests are passed
        return 'passed';
    }

    /**
     * Check if the writer is fully approved.
     */
    public function isApproved()
    {
        return $this->qualificationTest &&
            $this->qualificationTest->questions_test_status === 'passed' &&
            $this->essayTest &&
            $this->essayTest->essay_test_status === 'passed';
    }


    /**
     * Get the profile image URL
     *
     * @return string|null
     */
    public function getProfileImageUrlAttribute()
    {
        if (!$this->profile_image) {
            return null;
        }

        return Storage::disk('public')->url($this->profile_image);
    }

    /**
     * Calculate the profile completion percentage
     *
     * @return int
     */
    public function getProfileCompletionPercentageAttribute()
    {
        $fields = [
            'bio',
            'profile_image',
            'education_level_id',
            'specialization_id',
            'hourly_rate'
        ];

        $relationships = [
            'subjects' => 1,
            'languages' => 1,
            'citationStyles' => 1
        ];

        $total = count($fields) + count($relationships);
        $completed = 0;

        // Check fields
        foreach ($fields as $field) {
            if (!empty($this->$field)) {
                $completed++;
            }
        }

        // Check relationships
        foreach ($relationships as $relation => $minCount) {
            if ($this->$relation()->count() >= $minCount) {
                $completed++;
            }
        }

        return round(($completed / $total) * 100);
    }

    /**
     * Check if the writer profile is complete enough to be marked as available
     *
     * @return bool
     */
    public function isProfileComplete()
    {
        return $this->getProfileCompletionPercentageAttribute() >= 80; // Require at least 80% completion
    }

    /**
     * Get a list of required fields that are missing
     *
     * @return array
     */
    public function getMissingRequiredFields()
    {
        $missingFields = [];

        $requiredFields = [
            'bio' => 'Bio',
            'education_level_id' => 'Education Level',
        ];

        $requiredRelationships = [
            'subjects' => 'Subjects',
            'languages' => 'Languages',
        ];

        foreach ($requiredFields as $field => $label) {
            if (empty($this->$field)) {
                $missingFields[] = $label;
            }
        }

        foreach ($requiredRelationships as $relation => $label) {
            if ($this->$relation()->count() === 0) {
                $missingFields[] = $label;
            }
        }

        return $missingFields;
    }

    /**
     * Check if the writer profile can be updated
     *
     * @return bool
     */
    public function canUpdateFullProfile()
    {
        return $this->isApproved();
    }

    /**
     * Check if the writer profile can update basic info only
     *
     * @return bool
     */
    public function canUpdateBasicProfile()
    {
        return true; // All writers can update basic info
    }


    public function paymentAccounts()
    {
        return $this->hasMany(WriterPaymentAccount::class);
    }

    public function primaryPaymentAccount()
    {
        return $this->hasOne(WriterPaymentAccount::class)->where('is_primary', true);
    }
}
