<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;

class WriterQualificationTest extends Model
{
    use HasFactory;

    /**
     * The attributes that are mass assignable.
     *
     * @var array<int, string>
     */
    protected $fillable = [
        'writer_profile_id',
        'questions_test_status',
        'questions_score',
        'questions_completed_at',
        'last_failed_at',
        'can_retry_after',
        // New fields for security
        'started_at',              // When the test was started (server time)
        'expected_end_at',         // When the test should end based on duration
        'last_activity_at',        // Last recorded activity during the test
        'time_extensions',         // Number of legitimate time extensions granted
        'session_token',           // Unique token for this test session
        'ip_address',              // IP address used during the test
        'user_agent',              // Browser/device information
        'is_time_violation',       // Flag for suspicious timing behavior
        'time_violation_notes',     // Notes about any timing violations
        'auto_processed_at',       // When the test was automatically processed
    ];

    protected $casts = [
        'questions_score' => 'integer',
        'questions_completed_at' => 'datetime',
        'last_failed_at' => 'datetime',
        'can_retry_after' => 'datetime',

        'started_at' => 'datetime',
        'expected_end_at' => 'datetime',
        'last_activity_at' => 'datetime',
        'time_extensions' => 'integer',
        'is_time_violation' => 'boolean',
        'auto_processed_at' => 'datetime',
    ];


    /**
     * Get the writer profile that owns the qualification test.
     */
    public function writerProfile()
    {
        return $this->belongsTo(WriterProfile::class);
    }

    /**
     * Check if the writer can take the multiple-choice test.
     */
    public function canTakeTest()
    {
        // Writer can take the test if they haven't started or if they failed but the waiting period is over
        if ($this->questions_test_status === 'not_started') {
            return true;
        }

        if ($this->questions_test_status === 'questions_failed' && $this->can_retry_after) {
            return now()->gte($this->can_retry_after);
        }

        return false;
    }

    /**
     * Start a new test session
     *
     * @param int $durationMinutes Test duration in minutes
     * @param string $ipAddress User's IP address
     * @param string $userAgent User's browser/device information
     * @return string Session token
     */
    public function startTestSession($durationMinutes, $ipAddress, $userAgent)
    {
        $now = now();
        $sessionToken = md5($this->writer_profile_id . $now->timestamp . rand(1000, 9999));

        $this->update([
            'questions_test_status' => 'questions_pending',
            'started_at' => $now,
            'expected_end_at' => $now->addMinutes($durationMinutes),
            'last_activity_at' => $now,
            'session_token' => $sessionToken,
            'ip_address' => $ipAddress,
            'user_agent' => $userAgent,
            'is_time_violation' => false,
            'time_violation_notes' => null,
            'time_extensions' => 0,
            'auto_processed_at' => null,
        ]);

        return $sessionToken;
    }

    /**
     * Check if the test is still within the allowed time window
     *
     * @return bool
     */
    public function isWithinTimeWindow()
    {
        if (!$this->started_at || !$this->expected_end_at) {
            return false;
        }

        return now()->lte($this->expected_end_at);
    }

    /**
     * Get remaining test time in seconds
     *
     * @return int Seconds remaining (0 if expired)
     */
    public function getRemainingTime()
    {
        if (!$this->started_at || !$this->expected_end_at) {
            return 0;
        }

        $remainingSeconds = now()->diffInSeconds($this->expected_end_at, false);
        return max(0, $remainingSeconds);
    }

    /**
     * Record user activity during the test
     *
     * @param string $sessionToken Current session token
     * @param string $ipAddress Current IP address
     * @return bool Whether the activity was recorded successfully
     */
    public function recordActivity($sessionToken, $ipAddress)
    {
        // Verify session token matches
        if ($this->session_token !== $sessionToken) {
            $this->flagTimeViolation('Session token mismatch');
            return false;
        }

        // Check for IP address changes
        if ($this->ip_address !== $ipAddress) {
            $this->flagTimeViolation('IP address changed during test');
        }

        $this->update([
            'last_activity_at' => now()
        ]);

        return true;
    }

    /**
     * Flag a time violation
     *
     * @param string $notes Notes about the violation
     */
    public function flagTimeViolation($notes)
    {
        $this->update([
            'is_time_violation' => true,
            'time_violation_notes' => $notes
        ]);
    }

    /**
     * Verify test submission timing
     *
     * @param string $sessionToken Session token for verification
     * @return bool Whether the submission timing is valid
     */
    public function verifySubmissionTiming($sessionToken)
    {
        // Verify session token
        if ($this->session_token !== $sessionToken) {
            $this->flagTimeViolation('Invalid session token on submission');
            return false;
        }

        // Check if test is still within time window
        if (!$this->isWithinTimeWindow()) {
            $this->flagTimeViolation('Submission after time expired');
            return false;
        }

        // Check for suspicious gaps in activity
        if ($this->last_activity_at && now()->diffInMinutes($this->last_activity_at) > 5) {
            $this->flagTimeViolation('Suspicious gap in activity before submission');
            // We still allow the submission but flag it
        }

        return true;
    }

    /**
     * Check if the test has expired without submission
     *
     * @return bool
     */
    public function hasExpiredWithoutSubmission()
    {
        return $this->questions_test_status === 'questions_pending' &&
            $this->expected_end_at &&
            now()->gt($this->expected_end_at);
    }

    /**
     * Mark test as expired
     *
     * @param string|null $notes Additional notes about expiration
     * @return bool
     */
    public function markAsExpired($notes = null)
    {
        // Set retry date to 3 months from now
        $retryDate = now()->addMonths(3);

        return $this->update([
            'questions_test_status' => 'questions_failed',
            'questions_score' => 0,
            'questions_completed_at' => now(),
            'time_violation_notes' => $notes ?? 'Test expired without submission',
            'auto_processed_at' => now(),
            'last_failed_at' => now(),
            'can_retry_after' => $retryDate
        ]);
    }

    /**
     * Process partial submission (some answers submitted but not completed)
     *
     * @param array $answers Array of answers that were submitted
     * @return bool
     */
    public function processPartialSubmission($answers = [])
    {
        // Calculate score based on submitted answers
        $score = 0;
        $totalQuestions = WriterTestQuestion::where('is_active', true)->count();

        if (!empty($answers)) {
            $correctCount = WriterTestResponse::where('user_id', $this->writerProfile->user_id)
                ->where('is_correct', true)
                ->count();

            if ($totalQuestions > 0) {
                $score = ($correctCount / $totalQuestions) * 100;
            }
        }

        // Determine if passed (90% or higher)
        $passed = $score >= 90;

        return $this->update([
            'questions_test_status' => $passed ? 'passed' : 'questions_failed',
            'questions_score' => $score,
            'questions_completed_at' => now(),
            'time_violation_notes' => 'Partial submission auto-processed after deadline',
            'auto_processed_at' => now(),
            'last_failed_at' => $passed ? null : now(),
            'can_retry_after' => $passed ? null : now()->addMonths(3)
        ]);
    }
}
