<?php

namespace App\Notifications;

use App\Models\Bid;
use App\Models\Order;
use App\Traits\DatabaseNotificationTrait;
use Illuminate\Bus\Queueable;
use Illuminate\Contracts\Queue\ShouldQueue;
use Illuminate\Notifications\Messages\MailMessage;
use Illuminate\Notifications\Notification;

class AdminBidPlacedNotification extends Notification implements ShouldQueue
{
    use Queueable, DatabaseNotificationTrait;

    protected $order;
    protected $bid;

    /**
     * Create a new notification instance.
     *
     * @param Order $order
     * @param Bid $bid
     */
    public function __construct(Order $order, Bid $bid)
    {
        $this->order = $order;
        $this->bid = $bid;
    }

    /**
     * Get the notification's delivery channels.
     *
     * @param object $notifiable
     * @return array
     */
    public function via(object $notifiable): array
    {
        return ['mail', 'database'];
    }

    /**
     * Get the mail representation of the notification.
     *
     * @param object $notifiable
     * @return MailMessage
     */
    public function toMail(object $notifiable): MailMessage
    {
        // Get writer information
        $writer = $this->bid->writer;
        $writerName = $writer->name;
        $writerEmail = $writer->email;
        
        // Format the bid amount with currency symbol
        $bidAmount = '$' . number_format($this->bid->bid_amount, 2);
        
        // Format completion time
        $completionTime = \Carbon\Carbon::parse($this->bid->completion_time)->format('F j, Y, g:i a');

        return (new MailMessage)
            ->subject('New Bid Placed - Order #' . $this->order->order_number)
            ->greeting('Hello ' . $notifiable->name . '!')
            ->line('A new bid has been placed on an order that requires your attention.')
            ->line('**Order Details:**')
            ->line('• Order Number: #' . $this->order->order_number)
            ->line('• Title: ' . $this->order->title)
            ->line('• Subject: ' . $this->order->subject)
            ->line('• Pages: ' . $this->order->pages)
            ->line('• Client: ' . $this->order->user->name . ' (' . $this->order->user->email . ')')
            ->line('')
            ->line('**Bid Details:**')
            ->line('• Writer: ' . $writerName . ' (' . $writerEmail . ')')
            ->line('• Bid Amount: ' . $bidAmount)
            ->line('• Completion Time: ' . $completionTime)
            ->line('• Comments: ' . ($this->bid->comments ?: 'No additional comments'))
            ->action('Review Order & Bids', url('/admin/orders/' . $this->order->id))
            ->line('You may want to monitor this order to ensure quality and compliance.')
            ->line('Thank you for your attention to this matter.');
    }

    /**
     * Get the array representation of the notification.
     *
     * @param object $notifiable
     * @return array
     */
    public function toArray(object $notifiable): array
    {
        // Get writer information
        $writer = $this->bid->writer;
        
        // Create review order action
        $action = $this->createAction(
            'Review Order & Bids',
            url('/admin/orders/' . $this->order->id),
            'GET',
            'primary'
        );

        // Return standardized notification data
        return $this->getDatabaseNotificationData(
            'admin_bid_placed',
            "New Bid on Order #{$this->order->order_number}",
            "A writer has placed a bid on order #{$this->order->order_number} ({$this->order->title}).",
            'gavel',
            'info',
            'medium',
            'order',
            [$action],
            now()->addDays(7), // Expires after 7 days
            [
                'order_id' => $this->order->id,
                'order_number' => $this->order->order_number,
                'title' => $this->order->title,
                'subject' => $this->order->subject,
                'pages' => $this->order->pages,
                'client_id' => $this->order->user_id,
                'client_name' => $this->order->user->name,
                'client_email' => $this->order->user->email,
                'bid_id' => $this->bid->id,
                'writer_id' => $this->bid->writer_id,
                'writer_name' => $writer->name,
                'writer_email' => $writer->email,
                'bid_amount' => $this->bid->bid_amount,
                'formatted_bid_amount' => '$' . number_format($this->bid->bid_amount, 2),
                'completion_time' => $this->bid->completion_time,
                'formatted_completion_time' => \Carbon\Carbon::parse($this->bid->completion_time)->format('F j, Y, g:i a'),
                'comments' => $this->bid->comments,
                'bid_placed_at' => now()->toISOString(),
            ]
        );
    }
} 