<?php

namespace App\Notifications;

use Illuminate\Notifications\Notifiable;

class AdminNotifiable
{
    use Notifiable;

    public $email;
    public $name = 'Admin';
    public $id; // For database notifications

    /**
     * Create a new admin notifiable instance.
     *
     * @param string $email
     * @param string|null $name
     */
    public function __construct(string $email, ?string $name = null)
    {
        $this->email = $email;
        if ($name) {
            $this->name = $name;
        }

        // Generate a consistent ID based on email for database notifications
        $this->id = md5($email);
    }

    /**
     * Route notifications for the mail channel.
     *
     * @return string
     */
    public function routeNotificationForMail()
    {
        return $this->email;
    }

    /**
     * Get the notification routing information for database notifications.
     *
     * @return array
     */
    public function routeNotificationForDatabase()
    {
        return [
            'id' => $this->id,
            'type' => 'admin',
            'email' => $this->email
        ];
    }

    /**
     * Create multiple admin notifiables from an array of emails.
     *
     * @param array $emails
     * @return array
     */
    public static function createMany(array $emails): array
    {
        return array_map(function ($email) {
            return new self($email);
        }, $emails);
    }

    /**
     * Get the admin notifiable for a specific admin type.
     *
     * @param string $type Type of admin (e.g., 'system', 'support', 'finance')
     * @return self
     */
    public static function forType(string $type): self
    {
        $adminEmails = config('notifications.admin_emails', []);
        $email = $adminEmails[$type] ?? $adminEmails['default'] ?? env('ADMIN_EMAIL');

        return new self($email, ucfirst($type) . ' Admin');
    }
}
