<?php

namespace App\Notifications;

use App\Models\Order;
use App\Traits\DatabaseNotificationTrait;
use Illuminate\Bus\Queueable;
use Illuminate\Notifications\Notification;
use Illuminate\Contracts\Queue\ShouldQueue;
use Illuminate\Notifications\Messages\MailMessage;

class AdminOrderPlacedNotification extends Notification implements ShouldQueue
{
    use Queueable, DatabaseNotificationTrait;

    protected $order;
    protected $clientName;

    /**
     * Create a new notification instance.
     *
     * @param Order $order
     * @param string $clientName
     */
    public function __construct(Order $order, string $clientName)
    {
        $this->order = $order;
        $this->clientName = $clientName;
    }

    /**
     * Get the notification's delivery channels.
     *
     * @param object $notifiable
     * @return array
     */
    public function via(object $notifiable): array
    {
        return ['mail', 'database'];
    }

    /**
     * Get the mail representation of the notification.
     *
     * @param object $notifiable
     * @return MailMessage
     */
    public function toMail(object $notifiable): MailMessage
    {
        $amount = $this->order->order_amount ?? 'N/A';
        $deadline = $this->order->deadline
            ? \Carbon\Carbon::parse($this->order->deadline)->format('Y-m-d H:i:s')
            : 'Not specified';

        return (new MailMessage)
            ->subject('New Order #' . $this->order->order_number . ' Placed')
            ->greeting('Hello Admin,')
            ->line("An order has been placed successfully for client: {$this->clientName}.")
            ->line("Order Number: {$this->order->order_number}")
            ->line("Order Title: {$this->order->title}")
            ->line("Order Pages: {$this->order->pages}")
            ->line("Academic Level: {$this->order->academic_level}")
            ->line("Subject: {$this->order->subject}")
            ->line("Deadline: {$deadline}")
            ->line("Amount: \${$amount}")
            ->action('View Order', url("/admin/orders/{$this->order->id}"))
            ->line('Please review this new order at your earliest convenience.')
            ->line('Thank you for using our platform!');
    }

    /**
     * Get the array representation of the notification.
     *
     * @param object $notifiable
     * @return array
     */
    public function toArray(object $notifiable): array
    {
        // Create view order action
        $action = $this->createAction(
            'View Order',
            url("/admin/orders/{$this->order->id}"),
            'GET',
            'primary'
        );

        // Return standardized notification data
        return $this->getDatabaseNotificationData(
            'order_placed',
            "New Order #{$this->order->order_number}",
            "A new order has been placed by {$this->clientName} for {$this->order->pages} pages.",
            'file-text',
            'success',
            'high',  // High priority for new orders
            'order',
            [$action],
            null,    // No expiration
            [
                'order_id' => $this->order->id,
                'order_number' => $this->order->order_number,
                'client_name' => $this->clientName,
                'order_title' => $this->order->title,
                'order_pages' => $this->order->pages,
                'order_amount' => $this->order->order_amount,
                'order_deadline' => $this->order->deadline,
                'order_subject' => $this->order->subject,
                'order_type' => $this->order->type_of_paper,
                'order_academic_level' => $this->order->academic_level,
                'order_status' => $this->order->order_status,
                'payment_status' => $this->order->payment_status,
            ]
        );
    }
}
