<?php

namespace App\Notifications;

use App\Models\OrderRevision;
use Illuminate\Bus\Queueable;
use Illuminate\Contracts\Queue\ShouldQueue;
use Illuminate\Notifications\Messages\MailMessage;
use Illuminate\Notifications\Notification;

class AdminRevisionRequestedNotification extends Notification implements ShouldQueue
{
    use Queueable;

    protected OrderRevision $revision;

    /**
     * Create a new notification instance.
     */
    public function __construct(OrderRevision $revision)
    {
        $this->revision = $revision;
    }

    /**
     * Get the notification's delivery channels.
     *
     * @return array<int, string>
     */
    public function via(object $notifiable): array
    {
        return ['mail', 'database'];
    }

    /**
     * Get the mail representation of the notification.
     */
    public function toMail(object $notifiable): MailMessage
    {
        $order = $this->revision->order;
        $client = $this->revision->requestedBy;

        return (new MailMessage)
            ->subject("New Revision Request - Order #{$order->order_number}")
            ->greeting("Hello {$notifiable->name},")
            ->line("A new revision has been requested by a client.")
            ->line("**Order Details:**")
            ->line("Order Number: {$order->order_number}")
            ->line("Client: {$client->name}")
            ->line("Revision Type: " . ucfirst($this->revision->revision_type))
            ->line("Revision Number: {$this->revision->revision_number}")
            ->line("**Client Feedback:**")
            ->line($this->revision->revision_reason)
            ->action('Review Revision Request', route('admin.revisions.show', $this->revision))
            ->line('Please review and assign this revision to an appropriate writer.')
            ->salutation('Best regards, AcademicScribe Team');
    }

    /**
     * Get the array representation of the notification.
     *
     * @return array<string, mixed>
     */
    public function toArray(object $notifiable): array
    {
        return [
            'type' => 'revision_requested',
            'revision_id' => $this->revision->id,
            'order_id' => $this->revision->order_id,
            'order_number' => $this->revision->order->order_number,
            'client_name' => $this->revision->requestedBy->name,
            'revision_type' => $this->revision->revision_type,
            'revision_number' => $this->revision->revision_number,
            'message' => "Client requested revision for Order #{$this->revision->order->order_number}",
            'action_url' => route('admin.revisions.show', $this->revision),
        ];
    }
} 