<?php

namespace App\Notifications;

use App\Models\Order;
use App\Models\OrderFile;
use Illuminate\Bus\Queueable;
use Illuminate\Contracts\Queue\ShouldQueue;
use Illuminate\Notifications\Messages\MailMessage;
use Illuminate\Notifications\Notification;

class AdminWorkCompletedNotification extends Notification implements ShouldQueue
{
    use Queueable;

    protected $order;
    protected $orderFile;

    /**
     * Create a new notification instance.
     */
    public function __construct(Order $order, OrderFile $orderFile)
    {
        $this->order = $order;
        $this->orderFile = $orderFile;
    }

    /**
     * Get the notification's delivery channels.
     *
     * @return array<int, string>
     */
    public function via(object $notifiable): array
    {
        return ['mail', 'database'];
    }

    /**
     * Get the mail representation of the notification.
     */
    public function toMail(object $notifiable): MailMessage
    {
        $orderNumber = $this->order->order_number;
        $fileName = $this->orderFile->file_name;
        $adminName = $this->orderFile->uploader_name;

        return (new MailMessage)
            ->subject("Work Completed - Order #{$orderNumber}")
            ->greeting("Hello {$notifiable->name},")
            ->line("Great news! Your order has been completed and is ready for review.")
            ->line("**Order Details:**")
            ->line("• Order Number: #{$orderNumber}")
            ->line("• Title: {$this->order->title}")
            ->line("• Completed File: {$fileName}")
            ->line("• Completed by: {$adminName}")
            ->line("• Completion Date: " . $this->orderFile->date_uploaded->format('F j, Y \a\t g:i A'))
            ->line("Your order has been moved to 'Under Review' status. Please review the completed work and let us know if you need any revisions or if you're satisfied with the result.")
            ->action('View Order Details', url("/orders/{$this->order->id}"))
            ->line("If you have any questions or need assistance, please don't hesitate to contact our support team.")
            ->line("Thank you for choosing our service!")
            ->salutation("Best regards,\nThe AcademicScribe Team");
    }

    /**
     * Get the array representation of the notification.
     *
     * @return array<string, mixed>
     */
    public function toArray(object $notifiable): array
    {
        return [
            'order_id' => $this->order->id,
            'order_number' => $this->order->order_number,
            'file_id' => $this->orderFile->id,
            'file_name' => $this->orderFile->file_name,
            'admin_name' => $this->orderFile->uploader_name,
            'completion_date' => $this->orderFile->date_uploaded,
            'message' => "Your order #{$this->order->order_number} has been completed and is ready for review.",
            'type' => 'admin_work_completed',
            'action_url' => "/orders/{$this->order->id}"
        ];
    }

    /**
     * Get the notification's database type.
     */
    public function toDatabase(object $notifiable): array
    {
        return [
            'order_id' => $this->order->id,
            'order_number' => $this->order->order_number,
            'file_id' => $this->orderFile->id,
            'file_name' => $this->orderFile->file_name,
            'admin_name' => $this->orderFile->uploader_name,
            'completion_date' => $this->orderFile->date_uploaded,
            'message' => "Your order #{$this->order->order_number} has been completed and is ready for review.",
            'type' => 'admin_work_completed',
            'action_url' => "/orders/{$this->order->id}"
        ];
    }
} 