<?php

namespace App\Notifications;

use App\Models\Bid;
use App\Models\Order;
use App\Traits\DatabaseNotificationTrait;
use Illuminate\Bus\Queueable;
use Illuminate\Contracts\Queue\ShouldQueue;
use Illuminate\Notifications\Messages\MailMessage;
use Illuminate\Notifications\Notification;

class BidAssignmentConfirmationNotification extends Notification implements ShouldQueue
{
    use Queueable, DatabaseNotificationTrait;

    /**
     * The order that had a bid accepted.
     *
     * @var \App\Models\Order
     */
    protected $order;

    /**
     * The bid that was accepted.
     *
     * @var \App\Models\Bid
     */
    protected $bid;

    /**
     * Create a new notification instance.
     *
     * @param \App\Models\Order $order
     * @param \App\Models\Bid $bid
     * @return void
     */
    public function __construct(Order $order, Bid $bid)
    {
        $this->order = $order;
        $this->bid = $bid;
    }

    /**
     * Get the notification's delivery channels.
     *
     * @param mixed $notifiable
     * @return array
     */
    public function via($notifiable)
    {
        return $notifiable->preferredChannels('order_writer_assigned', ['database', 'mail']);
    }

    /**
     * Get the mail representation of the notification.
     *
     * @param mixed $notifiable
     * @return \Illuminate\Notifications\Messages\MailMessage
     */
    public function toMail($notifiable)
    {
        // Get writer nickname or safe identifier
        $writerName = $this->bid->writer && $this->bid->writer->writerProfile && $this->bid->writer->writerProfile->nickname
            ? $this->bid->writer->writerProfile->nickname
            : "Writer #" . $this->bid->writer_id;

        // Format currency for bid amount
        $bidAmount = '$' . number_format($this->bid->bid_amount, 2);

        return (new MailMessage)
            ->subject('Writer Assigned to Your Order #' . $this->order->order_number)
            ->greeting('Hello ' . $notifiable->name . ',')
            ->line('Great news! You have successfully assigned a writer to your order.')
            ->line('**Order Details:**')
            ->line('- Order #: ' . $this->order->order_number)
            ->line('- Title: ' . $this->order->title)
            ->line('- Writer: ' . $writerName)
            ->line('- Bid Amount: ' . $bidAmount)
            ->line('- Expected Completion: ' . date('M j, Y, g:i A', strtotime($this->bid->completion_time)))
            ->line('Your writer has been notified and will begin working on your order promptly.')
            ->action('View Order Details', url('/orders/' . $this->order->id))
            ->line('Thank you for using our platform!');
    }

    /**
     * Get the array representation of the notification.
     *
     * @param mixed $notifiable
     * @return array
     */
    public function toDatabase($notifiable)
    {
        // Get writer nickname or safe identifier
        $writerName = $this->bid->writer && $this->bid->writer->writerProfile && $this->bid->writer->writerProfile->nickname
            ? $this->bid->writer->writerProfile->nickname
            : "Writer #" . $this->bid->writer_id;

        return $this->getDatabaseNotificationData(
            'order_writer_assigned',
            'Writer Assigned to Order #' . $this->order->order_number,
            'You have successfully assigned ' . $writerName . ' to your order.',
            'check-circle',
            'success',
            'medium',
            'order',
            [
                $this->createAction(
                    'View Order',
                    url('/orders/' . $this->order->id),
                    'GET',
                    'primary'
                )
            ],
            null,
            [
                'order_id' => $this->order->id,
                'order_number' => $this->order->order_number,
                'bid_id' => $this->bid->id,
                'writer_id' => $this->bid->writer_id,
                'writer_name' => $writerName,
                'bid_amount' => $this->bid->bid_amount,
                'completion_time' => $this->bid->completion_time
            ]
        );
    }
}
