<?php

namespace App\Notifications;

use App\Models\Order;
use App\Traits\DatabaseNotificationTrait;
use Illuminate\Bus\Queueable;
use Illuminate\Contracts\Queue\ShouldQueue;
use Illuminate\Notifications\Messages\MailMessage;
use Illuminate\Notifications\Notification;

class BidRejectedNotification extends Notification implements ShouldQueue
{
    use Queueable, DatabaseNotificationTrait;

    protected $order;

    /**
     * Create a new notification instance.
     *
     * @param Order $order
     */
    public function __construct(Order $order)
    {
        $this->order = $order;
    }

    /**
     * Get the notification's delivery channels.
     *
     * @param object $notifiable
     * @return array
     */
    public function via(object $notifiable): array
    {
        return $notifiable->preferredChannels('bid_rejected', ['mail', 'database']);
    }

    /**
     * Get the mail representation of the notification.
     *
     * @param object $notifiable
     * @return MailMessage
     */
    public function toMail(object $notifiable): MailMessage
    {
        return (new MailMessage)
            ->subject('Bid Status Update: Order #' . $this->order->order_number)
            ->greeting('Hello ' . $notifiable->name . '!')
            ->line('We regret to inform you that your bid for Order #' . $this->order->order_number . ' was not selected.')
            ->line('Title: ' . $this->order->title)
            ->line('Thank you for your interest. We encourage you to bid on other available orders.')
            ->action('View Available Orders', url('/writer/orders'))
            ->line('Thank you for being part of our writing team!');
    }

    /**
     * Get the array representation of the notification.
     *
     * @param object $notifiable
     * @return array
     */
    public function toArray(object $notifiable): array
    {
        // Create view available orders action
        $viewOrdersAction = $this->createAction(
            'Find New Orders',
            url('/writer/orders'),
            'GET',
            'primary'
        );

        // Create view specific order action
        $viewOrderAction = $this->createAction(
            'View Order Details',
            url('/writer/orders/' . $this->order->id),
            'GET',
            'secondary'
        );

        // Return standardized notification data
        return $this->getDatabaseNotificationData(
            'bid_rejected',
            "Bid Not Selected for Order #{$this->order->order_number}",
            "Your bid for Order #{$this->order->order_number} ({$this->order->title}) was not selected.",
            'x-circle',
            'warning',
            'medium',
            'order',
            [$viewOrdersAction, $viewOrderAction],
            now()->addDays(5), // Expires after 5 days
            [
                'order_id' => $this->order->id,
                'order_number' => $this->order->order_number,
                'title' => $this->order->title,
                'subject' => $this->order->subject,
                'pages' => $this->order->pages,
                'rejected_at' => now()->toISOString(),
                'reason' => 'Another writer was selected for this order.'
            ]
        );
    }
}
