<?php

namespace App\Notifications;

use App\Models\Order;
use App\Traits\DatabaseNotificationTrait;
use Illuminate\Bus\Queueable;
use Illuminate\Notifications\Notification;
use Illuminate\Contracts\Queue\ShouldQueue;
use Illuminate\Notifications\Messages\MailMessage;

class ClientOrderPlacedNotification extends Notification implements ShouldQueue
{
    use Queueable, DatabaseNotificationTrait;

    protected $order;
    protected $adminName;

    /**
     * Create a new notification instance.
     *
     * @param Order $order
     * @param string|null $adminName
     */
    public function __construct(Order $order, ?string $adminName = 'Admin')
    {
        $this->order = $order;
        $this->adminName = $adminName;
    }

    /**
     * Get the notification's delivery channels.
     *
     * @param object $notifiable
     * @return array
     */
    public function via(object $notifiable): array
    {
        return ['mail', 'database'];
    }

    /**
     * Get the mail representation of the notification.
     *
     * @param object $notifiable
     * @return MailMessage
     */
    public function toMail(object $notifiable): MailMessage
    {
        $amount = $this->order->order_amount ?? 'N/A';
        $deadline = $this->order->deadline
            ? \Carbon\Carbon::parse($this->order->deadline)->format('F j, Y, g:i a')
            : 'Not specified';

        return (new MailMessage)
            ->subject('Order #' . $this->order->order_number . ' Placed on Your Behalf')
            ->greeting('Hello ' . $notifiable->name . ',')
            ->line("An order has been placed on your behalf by {$this->adminName}.")
            ->line("Order Number: {$this->order->order_number}")
            ->line("Order Title: {$this->order->title}")
            ->line("Order Pages: {$this->order->pages}")
            ->line("Academic Level: {$this->order->academic_level}")
            ->line("Subject: {$this->order->subject}")
            ->line("Paper Type: {$this->order->type_of_paper}")
            ->line("Deadline: {$deadline}")
            ->line("Total Amount: \${$amount}")
            ->action('View Order', url("/orders/{$this->order->id}"))
            ->line('If you have any questions about this order, please contact our support team.')
            ->line('Thank you for using our platform!');
    }

    /**
     * Get the array representation of the notification.
     *
     * @param object $notifiable
     * @return array
     */
    public function toArray(object $notifiable): array
    {
        // Create view order action
        $viewOrderAction = $this->createAction(
            'View Order',
            url("/orders/{$this->order->id}"),
            'GET',
            'primary'
        );

        // Create contact support action
        $contactSupportAction = $this->createAction(
            'Contact Support',
            url("/support/tickets/create?order_id={$this->order->id}"),
            'GET',
            'secondary'
        );

        // Return standardized notification data
        return $this->getDatabaseNotificationData(
            'order_placed_by_admin',
            "Order #{$this->order->order_number} Placed on Your Behalf",
            "An order has been placed on your behalf by {$this->adminName}. Order: {$this->order->title} ({$this->order->pages} pages).",
            'file-text',
            'info',
            'high',
            'order',
            [$viewOrderAction, $contactSupportAction],
            now()->addDays(30), // Expires after 30 days
            [
                'order_id' => $this->order->id,
                'order_number' => $this->order->order_number,
                'title' => $this->order->title,
                'pages' => $this->order->pages,
                'admin_name' => $this->adminName,
                'academic_level' => $this->order->academic_level,
                'subject' => $this->order->subject,
                'type_of_paper' => $this->order->type_of_paper,
                'deadline' => $this->order->deadline,
                'amount' => $this->order->order_amount,
                'created_at' => now()->toISOString(),
            ]
        );
    }
}
