<?php

namespace App\Notifications;

use App\Models\ContactMessage;
use App\Traits\DatabaseNotificationTrait;
use Illuminate\Bus\Queueable;
use Illuminate\Contracts\Queue\ShouldQueue;
use Illuminate\Notifications\Messages\MailMessage;
use Illuminate\Notifications\Notification;

class ContactMessageNotification extends Notification implements ShouldQueue
{
    use Queueable, DatabaseNotificationTrait;

    protected $contactMessage;

    /**
     * Create a new notification instance.
     */
    public function __construct(ContactMessage $contactMessage)
    {
        $this->contactMessage = $contactMessage;
    }

    /**
     * Get the notification's delivery channels.
     */
    public function via($notifiable): array
    {
        return ['mail'];
    }

    /**
     * Get the mail representation of the notification.
     */
    public function toMail($notifiable): MailMessage
    {
        return (new MailMessage)
            ->subject('Thank you for contacting Academic Scribe')
            ->greeting('Hello ' . $this->contactMessage->name . ',')
            ->line('Thank you for reaching out to us. We have received your message and will get back to you as soon as possible.')
            ->line('**Your Message Details:**')
            ->line('**Subject:** ' . $this->contactMessage->subject)
            ->line('**Message:** ' . $this->contactMessage->message)
            ->line('**Submitted on:** ' . $this->contactMessage->created_at->format('F j, Y \a\t g:i A'))
            ->line('We typically respond within 24 hours during business days.')
            ->line('If you have any urgent questions, please don\'t hesitate to contact us directly.')
            ->action('Visit Our Website', url('/'))
            ->line('Thank you for choosing Academic Scribe!')
            ->salutation('Best regards,')
            ->salutation('The Academic Scribe Team');
    }

    /**
     * Get the array representation of the notification.
     */
    public function toArray($notifiable): array
    {
        return [
            'contact_message_id' => $this->contactMessage->id,
            'name' => $this->contactMessage->name,
            'email' => $this->contactMessage->email,
            'subject' => $this->contactMessage->subject,
            'submitted_at' => $this->contactMessage->created_at,
        ];
    }
}
