<?php

namespace App\Notifications;

use Illuminate\Bus\Queueable;
use Illuminate\Contracts\Queue\ShouldQueue;
use Illuminate\Notifications\Messages\MailMessage;
use Illuminate\Notifications\Notification;
use App\Models\Conversation;
use App\Models\User;

class ConversationStatusNotification extends Notification implements ShouldQueue
{
    use Queueable;

    public Conversation $conversation;
    public string $oldStatus;
    public string $newStatus;
    public User $changedBy;
    public bool $isAutomatic;

    /**
     * Create a new notification instance.
     */
    public function __construct(
        Conversation $conversation,
        string $oldStatus,
        string $newStatus,
        User $changedBy,
        bool $isAutomatic = false
    ) {
        $this->conversation = $conversation;
        $this->oldStatus = $oldStatus;
        $this->newStatus = $newStatus;
        $this->changedBy = $changedBy;
        $this->isAutomatic = $isAutomatic;
    }

    /**
     * Get the notification's delivery channels.
     *
     * @return array<int, string>
     */
    public function via(object $notifiable): array
    {
        $channels = ['database'];
        
        // Add email if enabled in config
        if (config('messaging.notifications.enabled')) {
            $eventKey = "conversation_{$this->newStatus}";
            if (config("messaging.notifications.events.{$eventKey}", false)) {
                $channels[] = 'mail';
            }
        }
        
        return $channels;
    }

    /**
     * Get the mail representation of the notification.
     */
    public function toMail(object $notifiable): MailMessage
    {
        $message = new MailMessage();
        
        switch ($this->newStatus) {
            case 'resolved':
                $message->subject('Conversation Resolved')
                        ->line($this->getStatusChangeMessage())
                        ->line('This conversation has been marked as resolved.')
                        ->line('If you need to reopen this conversation, you can do so within 48 hours.');
                break;
                
            case 'reopened':
                $message->subject('Conversation Reopened')
                        ->line($this->getStatusChangeMessage())
                        ->line('This conversation has been reopened and is now active again.');
                break;
                
            case 'escalated':
                $message->subject('Conversation Escalated')
                        ->line($this->getStatusChangeMessage())
                        ->line('This conversation has been escalated for priority attention.');
                break;
                
            case 'archived':
                $message->subject('Conversation Archived')
                        ->line($this->getStatusChangeMessage())
                        ->line('This conversation has been archived and is no longer active.');
                break;
                
            default:
                $message->subject('Conversation Status Changed')
                        ->line($this->getStatusChangeMessage());
        }
        
        $message->action('View Conversation', route('messaging.conversations', ['id' => $this->conversation->id]))
                ->line('Thank you for using our messaging system!');
                
        return $message;
    }

    /**
     * Get the array representation of the notification.
     *
     * @return array<string, mixed>
     */
    public function toArray(object $notifiable): array
    {
        return [
            'conversation_id' => $this->conversation->id,
            'old_status' => $this->oldStatus,
            'new_status' => $this->newStatus,
            'changed_by' => [
                'id' => $this->changedBy->id,
                'name' => $this->changedBy->name,
                'type' => $this->changedBy->user_type,
            ],
            'is_automatic' => $this->isAutomatic,
            'message' => $this->getStatusChangeMessage(),
            'action_url' => route('messaging.conversations', ['id' => $this->conversation->id]),
        ];
    }

    /**
     * Get the status change message.
     */
    private function getStatusChangeMessage(): string
    {
        $otherParticipants = $this->conversation->participants
            ->where('id', '!=', $this->changedBy->id)
            ->pluck('name')
            ->join(', ');
            
        $actionBy = $this->isAutomatic ? 'automatically' : "by {$this->changedBy->name}";
        
        return "Your conversation with {$otherParticipants} has been {$this->newStatus} {$actionBy}.";
    }
}
