<?php

namespace App\Notifications;

use App\Models\WriterEssayTest;
use App\Traits\DatabaseNotificationTrait;
use Illuminate\Bus\Queueable;
use Illuminate\Contracts\Queue\ShouldQueue;
use Illuminate\Notifications\Messages\MailMessage;
use Illuminate\Notifications\Notification;

class EssayTestExpiredNotification extends Notification implements ShouldQueue
{
    use Queueable, DatabaseNotificationTrait;

    protected $essayTest;
    protected $essayTopic;

    /**
     * Create a new notification instance.
     *
     * @param WriterEssayTest $essayTest
     * @param string $essayTopic
     */
    public function __construct(WriterEssayTest $essayTest, string $essayTopic)
    {
        $this->essayTest = $essayTest;
        $this->essayTopic = $essayTopic;
    }

    /**
     * Get the notification's delivery channels.
     *
     * @param object $notifiable
     * @return array<int, string>
     */
    public function via(object $notifiable): array
    {
        return ['mail', 'database'];
    }

    /**
     * Get the mail representation of the notification.
     *
     * @param object $notifiable
     * @return MailMessage
     */
    public function toMail(object $notifiable): MailMessage
    {
        // Calculate retry date
        $retryDate = now()->addMonths(3)->format('F j, Y');

        return (new MailMessage)
            ->subject('Essay Test Time Expired')
            ->greeting('Hello ' . $notifiable->name . ',')
            ->line('Your essay test time has expired without a submission.')
            ->line('Topic: ' . $this->essayTopic)
            ->line('You will need to wait 3 months before you can attempt the essay test again.')
            ->line('You can retry the test after: ' . $retryDate)
            ->action('View Test Status', url(route('writer.essay.instructions')))
            ->line('We encourage you to use this time to practice your writing skills and prepare for the next attempt.')
            ->line('If you believe this was an error or you experienced technical difficulties, please contact our support team.');
    }

    /**
     * Get the array representation of the notification.
     *
     * @param object $notifiable
     * @return array<string, mixed>
     */
    public function toArray(object $notifiable): array
    {
        // Calculate retry date
        $retryDate = now()->addMonths(3);

        // Create view instructions action
        $instructionsAction = $this->createAction(
            'View Test Status',
            url(route('writer.essay.instructions')),
            'GET',
            'primary'
        );

        // Create contact support action
        $supportAction = $this->createAction(
            'Contact Support',
            url('/writer/support'),
            'GET',
            'secondary'
        );

        // Return standardized notification data
        return $this->getDatabaseNotificationData(
            'essay_test_expired',
            'Essay Test Time Expired',
            'Your essay test time has expired without a submission. You can retry the test after ' . $retryDate->format('F j, Y') . '.',
            'clock',
            'danger',
            'high',
            'qualification',
            [$instructionsAction, $supportAction],
            $retryDate, // Expires when they can retry
            [
                'essay_test_id' => $this->essayTest->id,
                'essay_topic' => $this->essayTopic,
                'expired_at' => now()->toISOString(),
                'can_retry_after' => $retryDate->toISOString(),
                'status' => 'expired',
            ]
        );
    }
}
