<?php

namespace App\Notifications;

use App\Traits\DatabaseNotificationTrait;
use Illuminate\Bus\Queueable;
use Illuminate\Contracts\Queue\ShouldQueue;
use Illuminate\Notifications\Messages\MailMessage;
use Illuminate\Notifications\Notification;

class NewEssaySubmissionNotification extends Notification implements ShouldQueue
{
    use Queueable, DatabaseNotificationTrait;

    protected $writerName;
    protected $writerEmail;
    protected $essayTopicId;
    protected $essayTopicTitle;
    protected $submissionTime;
    protected $isAutoSubmitted;
    protected $essayId;
    protected $writerId;

    /**
     * Create a new notification instance.
     *
     * @param string $writerName
     * @param string $writerEmail
     * @param int $essayTopicId
     * @param string $essayTopicTitle
     * @param \DateTime $submissionTime
     * @param bool $isAutoSubmitted
     * @param int|null $essayId
     * @param int|null $writerId
     */
    public function __construct(
        string $writerName,
        string $writerEmail,
        int $essayTopicId,
        string $essayTopicTitle,
        \DateTime $submissionTime,
        bool $isAutoSubmitted = false,
        ?int $essayId = null,
        ?int $writerId = null
    ) {
        $this->writerName = $writerName;
        $this->writerEmail = $writerEmail;
        $this->essayTopicId = $essayTopicId;
        $this->essayTopicTitle = $essayTopicTitle;
        $this->submissionTime = $submissionTime;
        $this->isAutoSubmitted = $isAutoSubmitted;
        $this->essayId = $essayId;
        $this->writerId = $writerId;
    }

    /**
     * Get the notification's delivery channels.
     *
     * @param object $notifiable
     * @return array<int, string>
     */
    public function via(object $notifiable): array
    {
        return ['mail', 'database'];
    }

    /**
     * Get the mail representation of the notification.
     *
     * @param object $notifiable
     * @return MailMessage
     */
    public function toMail(object $notifiable): MailMessage
    {
        // Get admin name if available or use default
        $adminName = $notifiable->name ?? 'Admin';

        $mailMessage = (new MailMessage)
            ->subject('New Essay Submission: ' . $this->writerName)
            ->greeting('Hello ' . $adminName . ',')
            ->line('A new essay has been submitted and is ready for review.')
            ->line('Writer: ' . $this->writerName . ' (' . $this->writerEmail . ')')
            ->line('Topic: ' . $this->essayTopicTitle)
            ->line('Submitted: ' . $this->submissionTime->format('F j, Y, g:i a'));

        if ($this->isAutoSubmitted) {
            $mailMessage->line('Note: This essay was auto-submitted when the time limit expired.');
        }

        return $mailMessage
            ->action('Review Essay', route('admin.essays.show', $this->essayTopicId))
            ->line('Please review this submission at your earliest convenience.');
    }

    /**
     * Get the array representation of the notification.
     *
     * @param object $notifiable
     * @return array<string, mixed>
     */
    public function toArray(object $notifiable): array
    {
        // Create primary action to review essay
        $reviewAction = $this->createAction(
            'Review Essay',
            url(route('admin.essays.show', $this->essayTopicId)),
            'GET',
            'primary'
        );

        // Create secondary action to view writer profile if writer ID is provided
        $actions = [$reviewAction];
        if ($this->writerId) {
            $actions[] = $this->createAction(
                'View Writer Profile',
                url(route('admin.users.show', $this->writerId)),
                'GET',
                'secondary'
            );
        }

        // Create notification title with auto-submission flag if needed
        $title = 'New Essay Submission: ' . $this->writerName;
        if ($this->isAutoSubmitted) {
            $title .= ' (Auto-Submitted)';
        }

        // Create notification message
        $message = 'A new essay on "' . $this->essayTopicTitle . '" has been submitted by ' .
            $this->writerName . ' on ' . $this->submissionTime->format('M j, Y, g:i a') . '.';

        // Add auto-submission note if applicable
        if ($this->isAutoSubmitted) {
            $message .= ' This was auto-submitted when the time limit expired.';
        }

        // Return standardized notification data
        return $this->getDatabaseNotificationData(
            'new_essay_submission',
            $title,
            $message,
            'file-text',
            $this->isAutoSubmitted ? 'warning' : 'info',
            'high',  // High priority for essay reviews
            'essay',
            $actions,
            now()->addDays(14),  // Expires after 14 days
            [
                'writer_id' => $this->writerId,
                'writer_name' => $this->writerName,
                'writer_email' => $this->writerEmail,
                'essay_id' => $this->essayId,
                'essay_topic_id' => $this->essayTopicId,
                'essay_topic_title' => $this->essayTopicTitle,
                'submission_time' => $this->submissionTime->format('Y-m-d H:i:s'),
                'is_auto_submitted' => $this->isAutoSubmitted,
                'needs_review' => true,
            ]
        );
    }
}
