<?php

namespace App\Notifications;

use App\Models\Order;
use App\Traits\DatabaseNotificationTrait;
use Illuminate\Bus\Queueable;
use Illuminate\Contracts\Queue\ShouldQueue;
use Illuminate\Notifications\Messages\MailMessage;
use Illuminate\Notifications\Notification;

class NewOrderAvailableForBiddingNotification extends Notification implements ShouldQueue
{
    use Queueable, DatabaseNotificationTrait;

    protected $order;

    /**
     * Create a new notification instance.
     *
     * @param Order $order
     */
    public function __construct(Order $order)
    {
        $this->order = $order;
    }

    /**
     * Get the notification's delivery channels.
     *
     * @param object $notifiable
     * @return array
     */
    public function via(object $notifiable): array
    {
        return ['mail', 'database'];
    }

    /**
     * Get the mail representation of the notification.
     *
     * @param object $notifiable
     * @return MailMessage
     */
    public function toMail(object $notifiable): MailMessage
    {
        // Format the deadline for display
        $deadline = $this->order->deadline
            ? \Carbon\Carbon::parse($this->order->deadline)->format('F j, Y, g:i a')
            : 'Not specified';

        // Format the writer amount with currency symbol
        $writerAmount = '$' . number_format($this->order->writer_amount, 2);

        return (new MailMessage)
            ->subject('New Order Available for Bidding - #' . $this->order->order_number)
            ->greeting('Hello ' . $notifiable->name . '!')
            ->line('A new order has just become available for bidding.')
            ->line('**Order Details:**')
            ->line('• Order Number: #' . $this->order->order_number)
            ->line('• Title: ' . $this->order->title)
            ->line('• Academic Level: ' . $this->order->academic_level)
            ->line('• Subject: ' . $this->order->subject)
            ->line('• Type of Paper: ' . $this->order->type_of_paper)
            ->line('• Urgency: ' . $this->order->urgency)
            ->line('• Pages: ' . $this->order->pages)
            ->line('• Writer Amount: ' . $writerAmount)
            ->line('• Client Deadline: ' . $deadline)
            ->line('• Writer Deadline: ' . \Carbon\Carbon::parse($this->order->writer_deadline)->format('F j, Y, g:i a'))
            ->line('')
            ->line('**Instructions:** ' . ($this->order->instructions ?: 'No specific instructions provided'))
            ->line('**Citation Styles:** ' . ($this->order->styles ?: 'Not specified'))
            ->line('**Sources Required:** ' . ($this->order->sources ?: 'Not specified'))
            ->action('View Order & Place Bid', url('/writer/orders/' . $this->order->id))
            ->line('This order is now open for bidding. Submit your competitive bid to win this project!')
            ->line('Good luck!');
    }

    /**
     * Get the array representation of the notification.
     *
     * @param object $notifiable
     * @return array
     */
    public function toArray(object $notifiable): array
    {
        // Create view order action
        $action = $this->createAction(
            'View Order & Place Bid',
            url('/writer/orders/' . $this->order->id),
            'GET',
            'primary'
        );

        // Return standardized notification data
        return $this->getDatabaseNotificationData(
            'new_order_available_for_bidding',
            "New Order #{$this->order->order_number} Available for Bidding",
            "A new order is available for bidding: {$this->order->title} ({$this->order->pages} pages, {$this->order->urgency} urgency)",
            'briefcase',
            'info',
            'high',
            'order',
            [$action],
            now()->addDays(3), // Expires after 3 days since bidding opportunities are time-sensitive
            [
                'order_id' => $this->order->id,
                'order_number' => $this->order->order_number,
                'title' => $this->order->title,
                'academic_level' => $this->order->academic_level,
                'subject' => $this->order->subject,
                'type_of_paper' => $this->order->type_of_paper,
                'urgency' => $this->order->urgency,
                'pages' => $this->order->pages,
                'writer_amount' => $this->order->writer_amount,
                'formatted_writer_amount' => '$' . number_format($this->order->writer_amount, 2),
                'deadline' => $this->order->deadline,
                'writer_deadline' => $this->order->writer_deadline,
                'instructions' => $this->order->instructions,
                'styles' => $this->order->styles,
                'sources' => $this->order->sources,
                'order_amount' => $this->order->order_amount,
                'net_amount' => $this->order->net_amount,
                'payment_status' => $this->order->payment_status,
                'order_status' => $this->order->order_status,
                'notification_sent_at' => now()->toISOString(),
            ]
        );
    }
} 