<?php

namespace App\Notifications;

use App\Models\Order;
use App\Traits\DatabaseNotificationTrait;
use Illuminate\Bus\Queueable;
use Illuminate\Notifications\Notification;
use Illuminate\Contracts\Queue\ShouldQueue;
use Illuminate\Notifications\Messages\MailMessage;

class NewOrderNotification extends Notification implements ShouldQueue
{
    use Queueable, DatabaseNotificationTrait;

    protected $order;
    protected $isAdmin;
    protected $adminName;

    /**
     * Create a new notification instance.
     *
     * @param Order $order
     * @param bool $isAdmin
     * @param string|null $adminName
     */
    public function __construct(Order $order, bool $isAdmin = false, ?string $adminName = null)
    {
        $this->order = $order;
        $this->isAdmin = $isAdmin;
        $this->adminName = $adminName;
    }

    /**
     * Get the notification's delivery channels.
     *
     * @param object $notifiable
     * @return array
     */
    public function via(object $notifiable): array
    {
        return ['mail', 'database'];
    }

    /**
     * Get the mail representation of the notification.
     *
     * @param object $notifiable
     * @return MailMessage
     */
    public function toMail(object $notifiable): MailMessage
    {
        if ($this->isAdmin) {
            return $this->toAdminMail($notifiable);
        }
        return $this->toUserMail($notifiable);
    }

    /**
     * Get the email for a client/user.
     *
     * @param object $notifiable
     * @return MailMessage
     */
    private function toUserMail(object $notifiable): MailMessage
    {
        // Format the deadline for display
        $deadline = $this->order->deadline
            ? \Carbon\Carbon::parse($this->order->deadline)->format('F j, Y, g:i a')
            : 'Not specified';

        return (new MailMessage)
            ->subject('Order Confirmation - #' . $this->order->order_number)
            ->greeting('Hello ' . $notifiable->name . ',')
            ->line('Thank you for placing your order with us.')
            ->line('Order Details:')
            ->line('Order Number: ' . $this->order->order_number)
            ->line('Title: ' . $this->order->title)
            ->line('Academic Level: ' . $this->order->academic_level)
            ->line('Subject: ' . $this->order->subject)
            ->line('Type of Paper: ' . $this->order->type_of_paper)
            ->line('Urgency: ' . $this->order->urgency)
            ->line('Instructions: ' . $this->order->instructions)
            ->line('Styles: ' . $this->order->styles)
            ->line('Sources: ' . $this->order->sources)
            ->line('PowerPoint Slides: ' . $this->order->powerpoint_slides)
            ->line('Pages: ' . $this->order->pages)
            ->line('Deadline: ' . $deadline)
            ->line('Total Amount: $' . $this->order->order_amount)
            ->line('Net Amount: $' . $this->order->net_amount)
            ->line('Discount: $' . $this->order->discount)
            ->action('View Order Details', url('/orders/' . $this->order->id))
            ->line('Thank you for choosing our services!');
    }

    /**
     * Get the email for an admin.
     *
     * @param object $notifiable
     * @return MailMessage
     */
    private function toAdminMail(object $notifiable): MailMessage
    {
        // Get admin name if available or use provided name or default
        $name = $notifiable->name ?? $this->adminName ?? 'Admin';

        // Format the deadline for display
        $deadline = $this->order->deadline
            ? \Carbon\Carbon::parse($this->order->deadline)->format('F j, Y, g:i a')
            : 'Not specified';

        // Get client name if available
        $clientName = $this->order->user ? $this->order->user->name : 'Client';

        return (new MailMessage)
            ->subject('New Order Received - #' . $this->order->order_number)
            ->greeting('Hello ' . $name . ',')
            ->line("A new order has been placed by {$clientName}.")
            ->line('Order Details:')
            ->line('Order Number: ' . $this->order->order_number)
            ->line('Title: ' . $this->order->title)
            ->line('Academic Level: ' . $this->order->academic_level)
            ->line('Subject: ' . $this->order->subject)
            ->line('Type of Paper: ' . $this->order->type_of_paper)
            ->line('Urgency: ' . $this->order->urgency)
            ->line('Instructions: ' . $this->order->instructions)
            ->line('Styles: ' . $this->order->styles)
            ->line('Sources: ' . $this->order->sources)
            ->line('PowerPoint Slides: ' . $this->order->powerpoint_slides)
            ->line('Pages: ' . $this->order->pages)
            ->line('Deadline: ' . $deadline)
            ->line('Total Amount: $' . $this->order->order_amount)
            ->line('Net Amount: $' . $this->order->net_amount)
            ->line('Discount: $' . $this->order->discount)
            ->action('View Order Details', url('/admin/orders/' . $this->order->id))
            ->line('Please review this order at your earliest convenience.');
    }

    /**
     * Get the array representation of the notification.
     *
     * @param object $notifiable
     * @return array
     */
    public function toArray(object $notifiable): array
    {
        // Create appropriate action URL based on user type
        $viewUrl = $this->isAdmin
            ? url('/admin/orders/' . $this->order->id)
            : url('/orders/' . $this->order->id);

        // Create view order action
        $viewOrderAction = $this->createAction(
            'View Order',
            $viewUrl,
            'GET',
            'primary'
        );

        // Create additional actions based on user type
        $actions = [$viewOrderAction];

        if ($this->isAdmin) {
            // Add admin-specific actions
            $actions[] = $this->createAction(
                'Assign Writer',
                url('/admin/orders/' . $this->order->id . '/assign'),
                'GET',
                'secondary'
            );
        } else {
            // Add client-specific actions
            if ($this->order->payment_status !== 'paid') {
                $actions[] = $this->createAction(
                    'Make Payment',
                    url('/orders/' . $this->order->id . '/payment'),
                    'GET',
                    'secondary'
                );
            }
        }

        // Determine title and message based on user type
        $title = $this->isAdmin
            ? "New Order #{$this->order->order_number} Received"
            : "Order #{$this->order->order_number} Confirmation";

        $message = $this->isAdmin
            ? "A new order has been placed. Order #{$this->order->order_number}: {$this->order->title} ({$this->order->pages} pages)"
            : "Your order has been placed successfully. Order #{$this->order->order_number}: {$this->order->title}";

        // Choose appropriate icon and color
        $icon = $this->isAdmin ? "inbox" : "check-circle";
        $color = $this->isAdmin ? "info" : "success";
        $priority = $this->isAdmin ? "high" : "medium";

        // Return standardized notification data
        return $this->getDatabaseNotificationData(
            'new_order',
            $title,
            $message,
            $icon,
            $color,
            $priority,
            'order',
            $actions,
            now()->addDays($this->isAdmin ? 14 : 30), // Different expiration for admin vs client
            [
                'order_id' => $this->order->id,
                'order_number' => $this->order->order_number,
                'title' => $this->order->title,
                'academic_level' => $this->order->academic_level,
                'subject' => $this->order->subject,
                'type_of_paper' => $this->order->type_of_paper,
                'urgency' => $this->order->urgency,
                'pages' => $this->order->pages,
                'deadline' => $this->order->deadline,
                'order_amount' => $this->order->order_amount,
                'net_amount' => $this->order->net_amount,
                'discount' => $this->order->discount,
                'payment_status' => $this->order->payment_status,
                'order_status' => $this->order->order_status,
                'is_admin_notification' => $this->isAdmin,
                'created_at' => now()->toISOString(),
            ]
        );
    }
}
