<?php

namespace App\Notifications;

use App\Models\Order;
use App\Traits\DatabaseNotificationTrait;
use Illuminate\Bus\Queueable;
use Illuminate\Contracts\Queue\ShouldQueue;
use Illuminate\Notifications\Messages\MailMessage;
use Illuminate\Notifications\Notification;

class OrderApprovalReminderNotification extends Notification implements ShouldQueue
{
    use Queueable, DatabaseNotificationTrait;

    protected $order;
    protected $hoursRemaining;

    /**
     * Create a new notification instance.
     *
     * @param Order $order
     * @param int $hoursRemaining
     */
    public function __construct(Order $order, int $hoursRemaining = 24)
    {
        $this->order = $order;
        $this->hoursRemaining = $hoursRemaining;
    }

    /**
     * Get the notification's delivery channels.
     *
     * @param object $notifiable
     * @return array<int, string>
     */
    public function via(object $notifiable): array
    {
        return $notifiable->preferredChannels('order_approval_reminder', ['database', 'mail']);
    }

    /**
     * Get the mail representation of the notification.
     *
     * @param object $notifiable
     * @return MailMessage
     */
    public function toMail(object $notifiable): MailMessage
    {
        $userName = $notifiable->name ?? 'Client';
        $writerName = $this->order->writer ? $this->order->writer->name : 'Your writer';

        $mailMessage = (new MailMessage)
            ->subject("Reminder: Review Your Order #{$this->order->order_number}")
            ->greeting("Hello {$userName},")
            ->line("This is a friendly reminder that your completed work for order #{$this->order->order_number} ({$this->order->title}) is ready for your review.")
            ->line("Writer: {$writerName}")
            ->line("Time remaining: {$this->hoursRemaining} hours");

        if ($this->hoursRemaining <= 24) {
            $mailMessage->line("⚠️ IMPORTANT: If you don't review this order within {$this->hoursRemaining} hours, it will be automatically approved.")
                ->line("Please take a moment to review the work and provide feedback if needed.");
        }

        $mailMessage->line("You can approve the work or request revisions if needed.")
            ->action('Review Order', url("/orders/{$this->order->id}/approve"))
            ->line('Thank you for your attention to this matter.');

        return $mailMessage;
    }

    /**
     * Get the array representation of the notification.
     *
     * @param object $notifiable
     * @return array<string, mixed>
     */
    public function toArray(object $notifiable): array
    {
        $writerName = $this->order->writer ? $this->order->writer->name : 'Your writer';

        // Create review order action
        $action = $this->createAction(
            'Review Order',
            url("/orders/{$this->order->id}/approve"),
            'GET',
            'primary'
        );

        $urgency = $this->hoursRemaining <= 24 ? 'high' : 'medium';
        $color = $this->hoursRemaining <= 24 ? 'warning' : 'info';

        return $this->getDatabaseNotificationData(
            'order_approval_reminder',
            "Order Review Reminder",
            "Your order #{$this->order->order_number} is ready for review. {$this->hoursRemaining} hours remaining before auto-approval.",
            'clock',
            $color,
            $urgency,
            'order',
            [$action],
            null,
            [
                'order_id' => $this->order->id,
                'order_number' => $this->order->order_number,
                'title' => $this->order->title,
                'writer_name' => $writerName,
                'hours_remaining' => $this->hoursRemaining,
                'reminder_sent_at' => now()->toISOString(),
                'auto_approval_at' => $this->order->client_review_started_at ? 
                    $this->order->client_review_started_at->addHours(48)->toISOString() : null
            ]
        );
    }
}
