<?php

namespace App\Notifications;

use App\Models\Order;
use App\Traits\DatabaseNotificationTrait;
use Carbon\Carbon;
use Illuminate\Bus\Queueable;
use Illuminate\Contracts\Queue\ShouldQueue;
use Illuminate\Notifications\Messages\MailMessage;
use Illuminate\Notifications\Notification;

class OrderDeadlineChange extends Notification implements ShouldQueue
{
    use Queueable, DatabaseNotificationTrait;

    protected $order;
    protected $oldDeadline;
    protected $changedBy;

    /**
     * Create a new notification instance.
     *
     * @param Order $order
     * @param string|null $oldDeadline
     * @param string|null $changedBy
     */
    public function __construct(Order $order, ?string $oldDeadline = null, ?string $changedBy = 'Admin')
    {
        $this->order = $order;
        $this->oldDeadline = $oldDeadline;
        $this->changedBy = $changedBy;
    }

    /**
     * Get the notification's delivery channels.
     *
     * @param object $notifiable
     * @return array<int, string>
     */
    public function via(object $notifiable): array
    {
        return ['mail', 'database'];
    }

    /**
     * Get the mail representation of the notification.
     *
     * @param object $notifiable
     * @return MailMessage
     */
    public function toMail(object $notifiable): MailMessage
    {
        // Format deadlines for display
        $newDeadline = Carbon::parse($this->order->deadline)->format('F j, Y, g:i a');

        // Format old deadline if available
        $oldDeadlineFormatted = $this->oldDeadline
            ? Carbon::parse($this->oldDeadline)->format('F j, Y, g:i a')
            : 'previous deadline';

        // Get client name if available
        $clientName = $notifiable->name ?? 'Client';

        return (new MailMessage)
            ->subject("Order #{$this->order->order_number} Deadline Changed")
            ->greeting("Hello {$clientName},")
            ->line("The deadline for your order #{$this->order->order_number} ({$this->order->title}) has been updated.")
            ->line("Changed by: {$this->changedBy}")
            ->line("Old deadline: {$oldDeadlineFormatted}")
            ->line("New deadline: {$newDeadline}")
            ->action('View Order Details', url("/orders/{$this->order->id}"))
            ->line('If you have any questions about this change, please contact our support team.')
            ->line('Thank you for using our service!');
    }

    /**
     * Get the array representation of the notification.
     *
     * @param object $notifiable
     * @return array<string, mixed>
     */
    public function toArray(object $notifiable): array
    {
        // Format deadlines for display
        $newDeadline = Carbon::parse($this->order->deadline)->format('F j, Y, g:i a');
        $oldDeadlineFormatted = $this->oldDeadline
            ? Carbon::parse($this->oldDeadline)->format('F j, Y, g:i a')
            : 'previous deadline';

        // Create view order action
        $viewOrderAction = $this->createAction(
            'View Order',
            url("/orders/{$this->order->id}"),
            'GET',
            'primary'
        );

        // Create contact support action
        $contactSupportAction = $this->createAction(
            'Contact Support',
            url("/support/tickets/create?order_id={$this->order->id}"),
            'GET',
            'secondary'
        );

        // Return standardized notification data
        return $this->getDatabaseNotificationData(
            'deadline_changed',
            "Order #{$this->order->order_number} Deadline Changed",
            "The deadline for your order has been changed from {$oldDeadlineFormatted} to {$newDeadline} by {$this->changedBy}.",
            'clock',
            'warning',
            'high',
            'order',
            [$viewOrderAction, $contactSupportAction],
            Carbon::parse($this->order->deadline), // Expires when the new deadline arrives
            [
                'order_id' => $this->order->id,
                'order_number' => $this->order->order_number,
                'order_title' => $this->order->title,
                'old_deadline' => $this->oldDeadline,
                'new_deadline' => $this->order->deadline,
                'changed_by' => $this->changedBy,
                'changed_at' => now()->toISOString(),
            ]
        );
    }
}
