<?php

namespace App\Notifications;

use App\Models\Order;
use App\Traits\DatabaseNotificationTrait;
use Illuminate\Bus\Queueable;
use Illuminate\Contracts\Queue\ShouldQueue;
use Illuminate\Notifications\Messages\MailMessage;
use Illuminate\Notifications\Notification;

class OrderStatusChange extends Notification implements ShouldQueue
{
    use Queueable, DatabaseNotificationTrait;

    protected $order;
    protected $previousStatus;
    protected $changedBy;

    /**
     * Create a new notification instance.
     *
     * @param Order $order
     * @param string|null $previousStatus
     * @param string|null $changedBy
     */
    public function __construct(Order $order, ?string $previousStatus = null, ?string $changedBy = 'Admin')
    {
        $this->order = $order;
        $this->previousStatus = $previousStatus;
        $this->changedBy = $changedBy;
    }

    /**
     * Get the notification's delivery channels.
     *
     * @param object $notifiable
     * @return array<int, string>
     */
    public function via(object $notifiable): array
    {
        // Return both mail and database channels
        return ['mail', 'database'];
    }

    /**
     * Get the mail representation of the notification.
     *
     * @param object $notifiable
     * @return MailMessage
     */
    public function toMail(object $notifiable): MailMessage
    {
        // Get user name - either from notifiable or from order relationship
        $userName = $notifiable->name ?? $this->order->user->name ?? 'Client';

        // Format the status for better readability
        $currentStatus = ucfirst(str_replace('_', ' ', $this->order->order_status));
        $fromStatus = $this->previousStatus
            ? ucfirst(str_replace('_', ' ', $this->previousStatus))
            : 'its previous status';

        return (new MailMessage)
            ->subject("Order #{$this->order->order_number} Status Changed")
            ->greeting("Hello {$userName},")
            ->line("The status of your order #{$this->order->order_number} ({$this->order->title}) has been changed from {$fromStatus} to {$currentStatus}.")
            ->line("Status changed by: {$this->changedBy}")
            ->line('Please check the order details for more information.')
            ->action('View Order Details', url("/orders/{$this->order->id}"))
            ->line('If you have any questions about this change, please contact our support team.')
            ->line('Thank you for using our service!');
    }

    /**
     * Get the array representation of the notification.
     *
     * @param object $notifiable
     * @return array<string, mixed>
     */
    public function toArray(object $notifiable): array
    {
        // Define status colors for visual indication
        $statusColors = [
            'pending' => 'gray',
            'bidding' => 'blue',
            'assigned' => 'purple',
            'in_progress' => 'yellow',
            'submitted' => 'indigo',
            'under_review' => 'orange',
            'approved' => 'green',
            'revision_requested' => 'red',
            'cancelled' => 'gray',
        ];

        // Get appropriate color based on status
        $color = $statusColors[$this->order->order_status] ?? 'default';

        // Define icon based on status
        $icons = [
            'pending' => 'clock',
            'bidding' => 'users',
            'assigned' => 'user-check',
            'in_progress' => 'edit-3',
            'submitted' => 'upload',
            'under_review' => 'eye',
            'approved' => 'check-circle',
            'revision_requested' => 'refresh-cw',
            'cancelled' => 'x-circle',
        ];

        $icon = $icons[$this->order->order_status] ?? 'file-text';

        // Create view order action
        $action = $this->createAction(
            'View Order',
            url("/orders/{$this->order->id}"),
            'GET',
            'primary'
        );

        // Return standardized notification data
        return $this->getDatabaseNotificationData(
            'order_status_change',
            "Order Status Updated",
            "Your order #{$this->order->order_number} status has been changed to {$this->order->getStatusDisplayName()}.",
            $icon,
            $color,
            'medium',
            'order',
            [$action],
            null,
            [
                'order_id' => $this->order->id,
                'order_number' => $this->order->order_number,
                'title' => $this->order->title,
                'status' => $this->order->order_status,
                'status_display_name' => $this->order->getStatusDisplayName(),
                'previous_status' => $this->previousStatus,
                'deadline' => $this->order->deadline,
                'changed_by' => $this->changedBy,
                'changed_at' => now()->toISOString(),
            ]
        );
    }
}
