<?php

namespace App\Notifications;

use App\Models\Order;
use App\Models\Payment;
use App\Traits\DatabaseNotificationTrait;
use Illuminate\Bus\Queueable;
use Illuminate\Contracts\Queue\ShouldQueue;
use Illuminate\Notifications\Messages\MailMessage;
use Illuminate\Notifications\Notification;

class PaymentRefundedNotification extends Notification implements ShouldQueue
{
    use Queueable, DatabaseNotificationTrait;

    protected $order;
    protected $payment;
    protected $refundAmount;
    protected $isFullRefund;
    protected $reason;

    /**
     * Create a new notification instance.
     *
     * @param Order $order
     * @param Payment $payment
     * @param float $refundAmount
     * @param bool $isFullRefund
     * @param string $reason
     */
    public function __construct(Order $order, Payment $payment, float $refundAmount, bool $isFullRefund, string $reason = '')
    {
        $this->order = $order;
        $this->payment = $payment;
        $this->refundAmount = $refundAmount;
        $this->isFullRefund = $isFullRefund;
        $this->reason = $reason;
    }

    /**
     * Get the notification's delivery channels.
     *
     * @param object $notifiable
     * @return array<int, string>
     */
    public function via(object $notifiable): array
    {
        return ['mail', 'database'];
    }

    /**
     * Get the mail representation of the notification.
     *
     * @param object $notifiable
     * @return MailMessage
     */
    public function toMail(object $notifiable): MailMessage
    {
        // Format amount for display
        $amount = number_format($this->refundAmount, 2);
        $currency = $this->order->currency ?? 'USD';

        // Format refund type
        $refundType = $this->isFullRefund ? 'Full' : 'Partial';

        // Create mail message
        $mailMessage = (new MailMessage)
            ->subject($refundType . ' Refund Processed - Order #' . $this->order->order_number)
            ->greeting('Hello ' . $notifiable->name . ',')
            ->line('A ' . strtolower($refundType) . ' refund of ' . $currency . ' ' . $amount . ' for Order #' . $this->order->order_number . ' has been processed.')
            ->line('Refund Details:')
            ->line('Amount: ' . $currency . ' ' . $amount)
            ->line('Payment Method: ' . ucfirst($this->payment->payment_method))
            ->line('Transaction ID: ' . ($this->payment->transaction_id ?? 'N/A'))
            ->line('Refund Date: ' . now()->format('F j, Y, g:i a'));

        // Add reason if provided
        if (!empty($this->reason)) {
            $mailMessage->line('Reason: ' . $this->reason);
        }

        return $mailMessage
            ->action('View Order Details', url('/orders/' . $this->order->id))
            ->line('If you have any questions about this refund, please contact our support team.');
    }

    /**
     * Get the array representation of the notification.
     *
     * @param object $notifiable
     * @return array<string, mixed>
     */
    public function toArray(object $notifiable): array
    {
        // Format amount for display
        $amount = number_format($this->refundAmount, 2);
        $currency = $this->order->currency ?? 'USD';

        // Create view order action
        $viewOrderAction = $this->createAction(
            'View Order',
            url('/orders/' . $this->order->id),
            'GET',
            'primary'
        );

        // Create contact support action
        $contactSupportAction = $this->createAction(
            'Contact Support',
            url('/support/tickets/create?order_id=' . $this->order->id),
            'GET',
            'secondary'
        );

        // Refund type
        $refundType = $this->isFullRefund ? 'full' : 'partial';

        // Return standardized notification data
        return $this->getDatabaseNotificationData(
            'payment_refunded',
            'Refund Processed',
            'A ' . $refundType . ' refund of ' . $currency . ' ' . $amount . ' for Order #' . $this->order->order_number . ' has been processed.',
            'credit-card',
            'info',
            'medium',
            'payment',
            [$viewOrderAction, $contactSupportAction],
            now()->addMonths(3), // Expires after 3 months
            [
                'order_id' => $this->order->id,
                'order_number' => $this->order->order_number,
                'payment_id' => $this->payment->id,
                'payment_method' => $this->payment->payment_method,
                'refund_amount' => $this->refundAmount,
                'is_full_refund' => $this->isFullRefund,
                'reason' => $this->reason,
                'refund_date' => now()->toISOString(),
                'order_title' => $this->order->title,
            ]
        );
    }
}
