<?php

namespace App\Notifications;

use App\Models\Order;
use App\Models\Payment;
use App\Traits\DatabaseNotificationTrait;
use Illuminate\Bus\Queueable;
use Illuminate\Contracts\Queue\ShouldQueue;
use Illuminate\Notifications\Messages\MailMessage;
use Illuminate\Notifications\Notification;

class PaymentSuccessfulNotification extends Notification implements ShouldQueue
{
    use Queueable, DatabaseNotificationTrait;

    protected $order;
    protected $payment;

    /**
     * Create a new notification instance.
     *
     * @param Order $order
     * @param Payment $payment
     */
    public function __construct(Order $order, Payment $payment)
    {
        $this->order = $order;
        $this->payment = $payment;
    }

    /**
     * Get the notification's delivery channels.
     *
     * @param object $notifiable
     * @return array<int, string>
     */
    public function via(object $notifiable): array
    {
        return ['mail', 'database'];
    }

    /**
     * Get the mail representation of the notification.
     *
     * @param object $notifiable
     * @return MailMessage
     */
    public function toMail(object $notifiable): MailMessage
    {
        // Format amount for display
        $amount = '$' . number_format($this->payment->amount, 2);

        // Format payment date
        $paymentDate = $this->payment->payment_date
            ? $this->payment->payment_date->format('F j, Y, g:i a')
            : now()->format('F j, Y, g:i a');

        // Format payment method for display
        $paymentMethod = ucfirst(str_replace('_', ' ', $this->payment->payment_method));

        return (new MailMessage)
            ->subject('Payment Successful - Order #' . $this->order->order_number)
            ->greeting('Hello ' . $notifiable->name . ',')
            ->line('Your payment for Order #' . $this->order->order_number . ' has been successfully processed.')
            ->line('Payment Details:')
            ->line('Amount: ' . $amount)
            ->line('Payment Method: ' . $paymentMethod)
            ->line('Transaction ID: ' . ($this->payment->transaction_id ?? 'N/A'))
            ->line('Payment Date: ' . $paymentDate)
            ->line('Your order is now being processed.')
            ->action('View Order Details', url('/orders/' . $this->order->id))
            ->line('Thank you for your payment!');
    }

    /**
     * Get the array representation of the notification.
     *
     * @param object $notifiable
     * @return array<string, mixed>
     */
    public function toArray(object $notifiable): array
    {
        // Format amount for display
        $amount = '$' . number_format($this->payment->amount, 2);

        // Create view order action
        $viewOrderAction = $this->createAction(
            'View Order',
            url('/orders/' . $this->order->id),
            'GET',
            'primary'
        );

        // Create view payment history action
        $viewPaymentHistoryAction = $this->createAction(
            'Payment History',
            url('/orders/' . $this->order->id . '/payment-history'),
            'GET',
            'secondary'
        );

        // Return standardized notification data
        return $this->getDatabaseNotificationData(
            'payment_successful',
            'Payment Successful',
            'Your payment of ' . $amount . ' for Order #' . $this->order->order_number . ' has been successfully processed.',
            'credit-card',
            'success',
            'high',
            'payment',
            [$viewOrderAction, $viewPaymentHistoryAction],
            now()->addMonths(3), // Expires after 3 months
            [
                'order_id' => $this->order->id,
                'order_number' => $this->order->order_number,
                'payment_id' => $this->payment->id,
                'payment_method' => $this->payment->payment_method,
                'amount' => $this->payment->amount,
                'transaction_id' => $this->payment->transaction_id,
                'payment_date' => $this->payment->payment_date ? $this->payment->payment_date->toISOString() : now()->toISOString(),
                'order_title' => $this->order->title,
            ]
        );
    }
}
