<?php

namespace App\Notifications;

use App\Models\OrderRevision;
use App\Traits\DatabaseNotificationTrait;
use Illuminate\Bus\Queueable;
use Illuminate\Contracts\Queue\ShouldQueue;
use Illuminate\Notifications\Messages\MailMessage;
use Illuminate\Notifications\Notification;

class RevisionQualityAssessedNotification extends Notification implements ShouldQueue
{
    use Queueable, DatabaseNotificationTrait;

    protected OrderRevision $revision;
    protected array $qualityData;

    /**
     * Create a new notification instance.
     */
    public function __construct(OrderRevision $revision, array $qualityData)
    {
        $this->revision = $revision;
        $this->qualityData = $qualityData;
    }

    /**
     * Get the notification's delivery channels.
     */
    public function via(object $notifiable): array
    {
        return $notifiable->preferredChannels('revision_quality_assessed', ['database', 'mail']);
    }

    /**
     * Get the mail representation of the notification.
     */
    public function toMail(object $notifiable): MailMessage
    {
        $order = $this->revision->order;
        $rating = $this->qualityData['quality_rating'];
        $paymentImpact = $this->qualityData['payment_impact'];
        $notes = $this->qualityData['quality_notes'] ?? 'No additional notes provided.';
        
        $mailMessage = (new MailMessage)
            ->subject("Quality Assessment Complete - Order #{$order->order_number}")
            ->greeting("Hello {$notifiable->name},")
            ->line("Your revision has been quality assessed by our team.")
            ->line("**Assessment Details:**")
            ->line("Order Number: {$order->order_number}")
            ->line("Order Title: {$order->title}")
            ->line("Revision Number: {$this->revision->revision_number}")
            ->line("Quality Rating: {$rating}/10")
            ->line("Payment Impact: $" . number_format($paymentImpact, 2))
            ->line("Assessment Notes: {$notes}");

        if ($paymentImpact > 0) {
            $mailMessage->line("🎉 Congratulations! You've earned a quality bonus.");
        } elseif ($paymentImpact < 0) {
            $mailMessage->line("⚠️ A quality penalty has been applied. Please review the feedback for future improvements.");
        } else {
            $mailMessage->line("✅ Good quality work. No payment impact.");
        }

        $mailMessage->action('View Revision Details', route('writer.revisions.show', $this->revision))
            ->line('Thank you for your work on this revision.')
            ->salutation('Best regards, AcademicScribe Team');

        return $mailMessage;
    }

    /**
     * Get the array representation of the notification.
     */
    public function toArray(object $notifiable): array
    {
        $order = $this->revision->order;
        $rating = $this->qualityData['quality_rating'];
        $paymentImpact = $this->qualityData['payment_impact'];

        // Create view revision action
        $action = $this->createAction(
            'View Revision',
            route('writer.revisions.show', $this->revision),
            'GET',
            $paymentImpact > 0 ? 'success' : ($paymentImpact < 0 ? 'warning' : 'info')
        );

        $icon = $paymentImpact > 0 ? 'star' : ($paymentImpact < 0 ? 'alert-triangle' : 'check-circle');
        $color = $paymentImpact > 0 ? 'success' : ($paymentImpact < 0 ? 'warning' : 'info');

        return $this->getDatabaseNotificationData(
            'revision_quality_assessed',
            "Quality Assessment Complete",
            "Your revision for order #{$order->order_number} has been quality assessed. Rating: {$rating}/10",
            $icon,
            $color,
            'high',
            'revision',
            [$action],
            null,
            [
                'order_id' => $order->id,
                'order_number' => $order->order_number,
                'title' => $order->title,
                'revision_id' => $this->revision->id,
                'revision_number' => $this->revision->revision_number,
                'quality_rating' => $rating,
                'quality_notes' => $this->qualityData['quality_notes'] ?? null,
                'payment_impact' => $paymentImpact,
                'quality_bonus' => $this->qualityData['quality_bonus'] ?? 0,
                'quality_penalty' => $this->qualityData['quality_penalty'] ?? 0,
                'assessed_at' => now()->toISOString()
            ]
        );
    }
} 