<?php

namespace App\Notifications;

use App\Models\OrderRevision;
use App\Traits\DatabaseNotificationTrait;
use Illuminate\Bus\Queueable;
use Illuminate\Contracts\Queue\ShouldQueue;
use Illuminate\Notifications\Messages\MailMessage;
use Illuminate\Notifications\Notification;

class RevisionRejectedNotification extends Notification implements ShouldQueue
{
    use Queueable, DatabaseNotificationTrait;

    protected OrderRevision $revision;
    protected string $rejectionReason;

    /**
     * Create a new notification instance.
     */
    public function __construct(OrderRevision $revision, string $rejectionReason)
    {
        $this->revision = $revision;
        $this->rejectionReason = $rejectionReason;
    }

    /**
     * Get the notification's delivery channels.
     */
    public function via(object $notifiable): array
    {
        return $notifiable->preferredChannels('revision_rejected', ['database', 'mail']);
    }

    /**
     * Get the mail representation of the notification.
     */
    public function toMail(object $notifiable): MailMessage
    {
        $order = $this->revision->order;
        
        return (new MailMessage)
            ->subject("Revision Rejected - Order #{$order->order_number}")
            ->greeting("Hello {$notifiable->name},")
            ->line("Your revision has been rejected and requires additional work.")
            ->line("**Order Details:**")
            ->line("Order Number: {$order->order_number}")
            ->line("Order Title: {$order->title}")
            ->line("Revision Number: {$this->revision->revision_number}")
            ->line("Revision Type: " . ucfirst($this->revision->revision_type))
            ->line("**Rejection Reason:**")
            ->line($this->rejectionReason)
            ->line("**Required Actions:**")
            ->line("Please review the feedback and make the necessary corrections.")
            ->line("The revision has been reset to 'requested' status.")
            ->action('View Revision Details', route('writer.revisions.show', $this->revision))
            ->line('Please address the issues and resubmit the revision.')
            ->salutation('Best regards, AcademicScribe Team');
    }

    /**
     * Get the array representation of the notification.
     */
    public function toArray(object $notifiable): array
    {
        $order = $this->revision->order;

        // Create view revision action
        $action = $this->createAction(
            'View Revision',
            route('writer.revisions.show', $this->revision),
            'GET',
            'warning'
        );

        return $this->getDatabaseNotificationData(
            'revision_rejected',
            "Revision Rejected",
            "Your revision for order #{$order->order_number} has been rejected and requires additional work.",
            'x-circle',
            'warning',
            'high',
            'revision',
            [$action],
            null,
            [
                'order_id' => $order->id,
                'order_number' => $order->order_number,
                'title' => $order->title,
                'revision_id' => $this->revision->id,
                'revision_number' => $this->revision->revision_number,
                'revision_type' => $this->revision->revision_type,
                'rejection_reason' => $this->rejectionReason,
                'rejected_at' => now()->toISOString()
            ]
        );
    }
} 