<?php

namespace App\Notifications;

use App\Models\Order;
use App\Traits\DatabaseNotificationTrait;
use Illuminate\Bus\Queueable;
use Illuminate\Contracts\Queue\ShouldQueue;
use Illuminate\Notifications\Messages\MailMessage;
use Illuminate\Notifications\Notification;

class RevisionRequestedNotification extends Notification implements ShouldQueue
{
    use Queueable, DatabaseNotificationTrait;

    protected $order;
    protected $revisionNotes;

    /**
     * Create a new notification instance.
     *
     * @param Order $order
     * @param string $revisionNotes
     */
    public function __construct(Order $order, string $revisionNotes)
    {
        $this->order = $order;
        $this->revisionNotes = $revisionNotes;
    }

    /**
     * Get the notification's delivery channels.
     *
     * @param object $notifiable
     * @return array<int, string>
     */
    public function via(object $notifiable): array
    {
        return $notifiable->preferredChannels('revision_requested', ['database', 'mail']);
    }

    /**
     * Get the mail representation of the notification.
     *
     * @param object $notifiable
     * @return MailMessage
     */
    public function toMail(object $notifiable): MailMessage
    {
        $userName = $notifiable->name ?? 'Writer';
        $clientName = $this->order->user ? $this->order->user->name : 'Your client';

        $mailMessage = (new MailMessage)
            ->subject("Revision Requested for Order #{$this->order->order_number}")
            ->greeting("Hello {$userName},")
            ->line("Your client has requested a revision for order #{$this->order->order_number} ({$this->order->title}).")
            ->line("Client: {$clientName}")
            ->line("Revision Notes:")
            ->line($this->revisionNotes)
            ->line("Please review the feedback and make the necessary changes.")
            ->line("You can start working on the revision immediately.")
            ->action('View Order Details', url("/writer/orders/{$this->order->id}"))
            ->line('Thank you for your attention to this matter.');

        return $mailMessage;
    }

    /**
     * Get the array representation of the notification.
     *
     * @param object $notifiable
     * @return array<string, mixed>
     */
    public function toArray(object $notifiable): array
    {
        $clientName = $this->order->user ? $this->order->user->name : 'Your client';

        // Create view order action
        $action = $this->createAction(
            'View Order',
            url("/writer/orders/{$this->order->id}"),
            'GET',
            'warning'
        );

        return $this->getDatabaseNotificationData(
            'revision_requested',
            "Revision Requested",
            "Your client has requested a revision for order #{$this->order->order_number}. Please review the feedback and make changes.",
            'refresh-cw',
            'warning',
            'high',
            'order',
            [$action],
            null,
            [
                'order_id' => $this->order->id,
                'order_number' => $this->order->order_number,
                'title' => $this->order->title,
                'client_name' => $clientName,
                'revision_notes' => $this->revisionNotes,
                'requested_at' => now()->toISOString(),
                'revision_count' => $this->order->revision_count ?? 0
            ]
        );
    }
}
