<?php

namespace App\Notifications;

use Illuminate\Bus\Queueable;
use Illuminate\Contracts\Queue\ShouldQueue;
use Illuminate\Notifications\Messages\MailMessage;
use Illuminate\Notifications\Notification;
use Illuminate\Notifications\Messages\DatabaseMessage;

class UrgentOrderNotification extends Notification implements ShouldQueue
{
    use Queueable;

    protected array $urgentOrders;
    protected string $userType;

    public function __construct(array $urgentOrders, string $userType)
    {
        $this->urgentOrders = $urgentOrders;
        $this->userType = $userType;
    }

    public function via($notifiable): array
    {
        return ['database', 'mail'];
    }

    public function toMail($notifiable): MailMessage
    {
        $subject = $this->userType === 'admin' 
            ? 'Urgent Orders Requiring Attention' 
            : 'You Have Urgent Orders Due Soon';

        $message = (new MailMessage)
            ->subject($subject)
            ->greeting('Hello ' . $notifiable->name . '!');

        if ($this->userType === 'admin') {
            $message->line('There are urgent orders that require immediate attention:');
        } else {
            $message->line('You have urgent orders that are due soon:');
        }

        // Add order details
        foreach ($this->urgentOrders as $urgencyLevel => $orders) {
            if (!empty($orders)) {
                $urgencyLabel = match ($urgencyLevel) {
                    'critical_12h' => 'Critical (Due within 12 hours)',
                    'urgent_24h' => 'Urgent (Due within 24 hours)',
                    'warning_48h' => 'Warning (Due within 48 hours)',
                    default => 'Urgent'
                };

                $message->line($urgencyLabel . ': ' . count($orders) . ' order(s)');
            }
        }

        $message->action('View Dashboard', url('/' . $this->userType . '/dashboard'))
            ->line('Please take immediate action on these orders.');

        return $message;
    }

    public function toDatabase($notifiable): array
    {
        $totalOrders = collect($this->urgentOrders)->flatten(1)->count();
        
        return [
            'title' => $this->userType === 'admin' 
                ? 'Urgent Orders Requiring Attention' 
                : 'Urgent Orders Due Soon',
            'message' => "You have {$totalOrders} urgent order(s) that require immediate attention.",
            'urgent_orders' => $this->urgentOrders,
            'user_type' => $this->userType,
            'total_count' => $totalOrders,
            'type' => 'urgent_orders'
        ];
    }

    public function toArray($notifiable): array
    {
        return [
            'urgent_orders' => $this->urgentOrders,
            'user_type' => $this->userType,
            'total_count' => collect($this->urgentOrders)->flatten(1)->count()
        ];
    }
} 