<?php

namespace App\Notifications;

use App\Models\Order;
use App\Models\OrderFile;
use App\Traits\DatabaseNotificationTrait;
use Illuminate\Bus\Queueable;
use Illuminate\Contracts\Queue\ShouldQueue;
use Illuminate\Notifications\Messages\MailMessage;
use Illuminate\Notifications\Notification;

class WorkReadyForReviewNotification extends Notification implements ShouldQueue
{
    use Queueable, DatabaseNotificationTrait;

    protected $order;
    protected $deliveredFiles;

    /**
     * Create a new notification instance.
     *
     * @param Order $order
     * @param array $deliveredFiles
     */
    public function __construct(Order $order, array $deliveredFiles = [])
    {
        $this->order = $order;
        $this->deliveredFiles = $deliveredFiles;
    }

    /**
     * Get the notification's delivery channels.
     *
     * @param object $notifiable
     * @return array<int, string>
     */
    public function via(object $notifiable): array
    {
        return $notifiable->preferredChannels('work_ready_for_review', ['database', 'mail']);
    }

    /**
     * Get the mail representation of the notification.
     *
     * @param object $notifiable
     * @return MailMessage
     */
    public function toMail(object $notifiable): MailMessage
    {
        $userName = $notifiable->name ?? 'Client';
        $writerName = $this->order->writer ? $this->order->writer->name : 'Your writer';
        $fileCount = count($this->deliveredFiles);

        $mailMessage = (new MailMessage)
            ->subject("Work Ready for Review - Order #{$this->order->order_number}")
            ->greeting("Hello {$userName},")
            ->line("Excellent news! Your completed work for order #{$this->order->order_number} ({$this->order->title}) is ready for your review.")
            ->line("Writer: {$writerName}")
            ->line("Files delivered: {$fileCount} file(s)");

        if (!empty($this->deliveredFiles)) {
            $mailMessage->line("Delivered files:");
            foreach ($this->deliveredFiles as $file) {
                $mailMessage->line("- {$file->file_name}");
            }
        }

        $mailMessage->line("Our admin team has reviewed the work and uploaded it to your account.")
            ->line("You have 48 hours to review and approve the work, or it will be automatically approved.")
            ->line("You can approve the work or request revisions if needed.")
            ->action('Review Work', url("/orders/{$this->order->id}/approve"))
            ->line('Thank you for choosing our service!');

        return $mailMessage;
    }

    /**
     * Get the array representation of the notification.
     *
     * @param object $notifiable
     * @return array<string, mixed>
     */
    public function toArray(object $notifiable): array
    {
        $writerName = $this->order->writer ? $this->order->writer->name : 'Your writer';
        $fileCount = count($this->deliveredFiles);

        // Create review work action
        $action = $this->createAction(
            'Review Work',
            url("/orders/{$this->order->id}/approve"),
            'GET',
            'primary'
        );

        return $this->getDatabaseNotificationData(
            'work_ready_for_review',
            "Work Ready for Review",
            "Your completed work for order #{$this->order->order_number} is ready for review. You have 48 hours to approve or request revisions.",
            'eye',
            'success',
            'high',
            'order',
            [$action],
            null,
            [
                'order_id' => $this->order->id,
                'order_number' => $this->order->order_number,
                'title' => $this->order->title,
                'writer_name' => $writerName,
                'file_count' => $fileCount,
                'ready_at' => now()->toISOString(),
                'deadline' => $this->order->client_review_started_at ? 
                    $this->order->client_review_started_at->addHours(48)->toISOString() : null,
                'files' => $this->deliveredFiles->map(function ($file) {
                    return [
                        'name' => $file->file_name,
                        'size' => $file->file_size,
                        'type' => $file->file_type
                    ];
                })->toArray()
            ]
        );
    }
}
