<?php

namespace App\Notifications;

use App\Models\Order;
use App\Models\OrderFile;
use App\Traits\DatabaseNotificationTrait;
use Illuminate\Bus\Queueable;
use Illuminate\Contracts\Queue\ShouldQueue;
use Illuminate\Notifications\Messages\MailMessage;
use Illuminate\Notifications\Notification;

class WorkSubmittedAdminNotification extends Notification implements ShouldQueue
{
    use Queueable, DatabaseNotificationTrait;

    protected $order;
    protected $uploadedFiles;

    /**
     * Create a new notification instance.
     *
     * @param Order $order
     * @param array $uploadedFiles
     */
    public function __construct(Order $order, array $uploadedFiles = [])
    {
        $this->order = $order;
        $this->uploadedFiles = $uploadedFiles;
    }

    /**
     * Get the notification's delivery channels.
     *
     * @param object $notifiable
     * @return array<int, string>
     */
    public function via(object $notifiable): array
    {
        return $notifiable->preferredChannels('work_submitted_admin', ['database', 'mail']);
    }

    /**
     * Get the mail representation of the notification.
     *
     * @param object $notifiable
     * @return MailMessage
     */
    public function toMail(object $notifiable): MailMessage
    {
        $writerName = $this->order->writer ? $this->order->writer->name : 'Unknown Writer';
        $clientName = $this->order->user ? $this->order->user->name : 'Unknown Client';
        $fileCount = count($this->uploadedFiles);
        $submissionTime = $this->order->work_submitted_at ? $this->order->work_submitted_at->format('M j, Y g:i A') : 'Just now';
        $deadline = $this->order->writer_deadline ? $this->order->writer_deadline->format('M j, Y g:i A') : 'Not set';
        $isLateSubmission = $this->order->work_submitted_at && $this->order->writer_deadline && 
                           $this->order->work_submitted_at->isAfter($this->order->writer_deadline);

        $mailMessage = (new MailMessage)
            ->subject("Work Submitted for Review - Order #{$this->order->order_number}")
            ->greeting("Hello Admin,")
            ->line("A writer has submitted completed work that requires your review.")
            ->line("Order Details:")
            ->line("- Order #: {$this->order->order_number}")
            ->line("- Title: {$this->order->title}")
            ->line("- Writer: {$writerName}")
            ->line("- Client: {$clientName}")
            ->line("- Submission Time: {$submissionTime}")
            ->line("- Writer Deadline: {$deadline}")
            ->line("- Files submitted: {$fileCount} file(s)");

        if ($isLateSubmission) {
            $hoursLate = $this->order->work_submitted_at->diffInHours($this->order->writer_deadline);
            $mailMessage->line("⚠️ LATE SUBMISSION: This work was submitted {$hoursLate} hour(s) after the deadline.")
                ->line("Please review for quality and assess any applicable penalties.");
        } else {
            $mailMessage->line("✅ ON-TIME SUBMISSION: Work was submitted before the deadline.");
        }

        if (!empty($this->uploadedFiles)) {
            $mailMessage->line("Submitted files:");
            foreach ($this->uploadedFiles as $file) {
                $mailMessage->line("- {$file->file_name} ({$file->file_size} bytes)");
            }
        }

        $mailMessage->line("Please review the work and upload it to the client's account.")
            ->line("The client will be notified once the work is ready for their review.")
            ->action('Review Order', url("/admin/orders/{$this->order->id}"))
            ->line('Thank you for your attention to this matter.');

        return $mailMessage;
    }

    /**
     * Get the array representation of the notification.
     *
     * @param object $notifiable
     * @return array<string, mixed>
     */
    public function toArray(object $notifiable): array
    {
        $writerName = $this->order->writer ? $this->order->writer->name : 'Unknown Writer';
        $clientName = $this->order->user ? $this->order->user->name : 'Unknown Client';
        $fileCount = count($this->uploadedFiles);
        $submissionTime = $this->order->work_submitted_at ? $this->order->work_submitted_at->format('M j, Y g:i A') : 'Just now';
        $deadline = $this->order->writer_deadline ? $this->order->writer_deadline->format('M j, Y g:i A') : 'Not set';
        $isLateSubmission = $this->order->work_submitted_at && $this->order->writer_deadline && 
                           $this->order->work_submitted_at->isAfter($this->order->writer_deadline);

        // Create review order action
        $action = $this->createAction(
            'Review Order',
            url("/admin/orders/{$this->order->id}"),
            'GET',
            'primary'
        );

        $statusMessage = $isLateSubmission 
            ? "{$writerName} has submitted LATE work for order #{$this->order->order_number}. Please review and assess penalties."
            : "{$writerName} has submitted work for order #{$this->order->order_number}. Please review and upload to client.";

        return $this->getDatabaseNotificationData(
            'work_submitted_admin',
            "Work Submitted for Review",
            $statusMessage,
            'eye',
            $isLateSubmission ? 'danger' : 'warning',
            'high',
            'order',
            [$action],
            null,
            [
                'order_id' => $this->order->id,
                'order_number' => $this->order->order_number,
                'title' => $this->order->title,
                'writer_name' => $writerName,
                'client_name' => $clientName,
                'file_count' => $fileCount,
                'submission_time' => $this->order->work_submitted_at ? $this->order->work_submitted_at->toISOString() : now()->toISOString(),
                'writer_deadline' => $this->order->writer_deadline ? $this->order->writer_deadline->toISOString() : null,
                'is_late_submission' => $isLateSubmission,
                'hours_late' => $isLateSubmission && $this->order->work_submitted_at && $this->order->writer_deadline ? 
                    $this->order->work_submitted_at->diffInHours($this->order->writer_deadline) : 0,
                'files' => $this->uploadedFiles->map(function ($file) {
                    return [
                        'name' => $file->file_name,
                        'size' => $file->file_size,
                        'type' => $file->file_type
                    ];
                })->toArray()
            ]
        );
    }
}
