<?php

namespace App\Notifications;

use Illuminate\Bus\Queueable;
use Illuminate\Contracts\Queue\ShouldQueue;
use Illuminate\Notifications\Messages\MailMessage;
use Illuminate\Notifications\Notification;

class WriterDisciplinaryAction extends Notification implements ShouldQueue
{
    use Queueable;

    protected $actionType;
    protected $actionDetails;
    protected $effectiveDate;
    protected $duration;
    protected $conditions;
    protected $reviewDate;

    /**
     * Create a new notification instance.
     */
    public function __construct(
        string $actionType,
        string $actionDetails,
        $effectiveDate = null,
        $duration = null,
        array $conditions = [],
        $reviewDate = null
    ) {
        $this->actionType = $actionType;
        $this->actionDetails = $actionDetails;
        $this->effectiveDate = $effectiveDate;
        $this->duration = $duration;
        $this->conditions = $conditions;
        $this->reviewDate = $reviewDate;
    }

    /**
     * Get the notification's delivery channels.
     *
     * @return array<int, string>
     */
    public function via(object $notifiable): array
    {
        return ['mail', 'database'];
    }

    /**
     * Get the mail representation of the notification.
     */
    public function toMail(object $notifiable): MailMessage
    {
        $actionTypes = [
            'warning' => 'Warning Issued',
            'probation' => 'Account Placed on Probation',
            'suspension' => 'Account Suspended',
            'restriction' => 'Account Restrictions Applied',
            'termination' => 'Account Termination Notice',
            'performance_plan' => 'Performance Improvement Plan',
            'other' => 'Disciplinary Action Applied'
        ];

        $actionTitle = $actionTypes[$this->actionType] ?? ucfirst($this->actionType);

        $message = (new MailMessage)
            ->subject('🚨 DISCIPLINARY ACTION - ' . $actionTitle)
            ->greeting('Hello ' . $notifiable->name . ',')
            ->line('**IMPORTANT: A disciplinary action has been applied to your account.**')
            ->line('**Action Details:**')
            ->line('• **Action Type:** ' . $actionTitle)
            ->line('• **Details:** ' . $this->actionDetails)
            ->line('• **Effective Date:** ' . ($this->effectiveDate ? $this->effectiveDate->format('M d, Y') : 'Immediate'));

        if ($this->duration) {
            $message->line('• **Duration:** ' . $this->duration);
        }

        if ($this->reviewDate) {
            $message->line('• **Review Date:** ' . $this->reviewDate->format('M d, Y'));
        }

        if (!empty($this->conditions)) {
            $message->line('**Conditions:**');
            foreach ($this->conditions as $condition) {
                $message->line('• ' . $condition);
            }
        }

        // Add specific information based on action type
        switch ($this->actionType) {
            case 'probation':
                $message->line('**Probation Status:**')
                    ->line('Your account is now under probationary review. During this period:')
                    ->line('1. New assignments may be limited or restricted')
                    ->line('2. All work will be subject to enhanced review')
                    ->line('3. Performance will be closely monitored')
                    ->line('4. Failure to meet conditions may result in suspension');
                break;

            case 'suspension':
                $message->line('**Suspension Details:**')
                    ->line('Your account has been suspended. During suspension:')
                    ->line('1. You cannot accept new assignments')
                    ->line('2. Existing orders may be reassigned')
                    ->line('3. Payments may be held until resolution')
                    ->line('4. Contact support immediately to discuss reinstatement');
                break;

            case 'restriction':
                $message->line('**Account Restrictions:**')
                    ->line('The following restrictions have been applied to your account:')
                    ->line('1. Limited access to certain order types')
                    ->line('2. Reduced assignment priority')
                    ->line('3. Enhanced review requirements')
                    ->line('4. Payment processing delays');
                break;

            case 'performance_plan':
                $message->line('**Performance Improvement Plan:**')
                    ->line('You are now under a Performance Improvement Plan (PIP):')
                    ->line('1. Specific performance goals have been set')
                    ->line('2. Regular progress reviews will be conducted')
                    ->line('3. Failure to meet goals may result in further action')
                    ->line('4. Success will result in plan completion and status restoration');
                break;
        }

        $message->line('**Immediate Actions Required:**')
            ->line('1. Review and acknowledge this disciplinary action')
            ->line('2. Contact support within 24 hours to discuss the action')
            ->line('3. Understand and comply with all conditions')
            ->line('4. Take steps to address underlying issues');

        if ($this->actionType !== 'termination') {
            $message->line('**Appeal Rights:**')
                ->line('You have the right to appeal this disciplinary action within 48 hours.')
                ->action('Submit Appeal', route('writer.disciplinary.appeal'))
                ->action('Contact Support', route('support.contact'));
        }

        $message->line('**Account Status:**')
            ->line('Your account status has been updated. Please review your dashboard for current restrictions and requirements.')
            ->salutation('Best regards, AcademicScribe Disciplinary Team');

        return $message;
    }

    /**
     * Get the array representation of the notification.
     *
     * @return array<string, mixed>
     */
    public function toArray(object $notifiable): array
    {
        return [
            'action_type' => $this->actionType,
            'action_details' => $this->actionDetails,
            'effective_date' => $this->effectiveDate,
            'duration' => $this->duration,
            'conditions' => $this->conditions,
            'review_date' => $this->reviewDate,
            'applied_at' => now(),
            'type' => 'writer_disciplinary_action',
            'message' => 'Disciplinary action applied: ' . ucfirst($this->actionType),
            'priority' => 'high',
        ];
    }

    /**
     * Get the notification's database type.
     */
    public function toDatabase(object $notifiable): array
    {
        return $this->toArray($notifiable);
    }
} 