<?php

namespace App\Notifications;

use App\Models\WriterFine;
use Illuminate\Bus\Queueable;
use Illuminate\Contracts\Queue\ShouldQueue;
use Illuminate\Notifications\Messages\MailMessage;
use Illuminate\Notifications\Notification;

class WriterFineAppealDecision extends Notification implements ShouldQueue
{
    use Queueable;

    protected $fine;
    protected $decision;
    protected $decisionNotes;
    protected $reviewedBy;

    /**
     * Create a new notification instance.
     */
    public function __construct(WriterFine $fine, string $decision, string $decisionNotes, $reviewedBy)
    {
        $this->fine = $fine;
        $this->decision = $decision;
        $this->decisionNotes = $decisionNotes;
        $this->reviewedBy = $reviewedBy;
    }

    /**
     * Get the notification's delivery channels.
     *
     * @return array<int, string>
     */
    public function via(object $notifiable): array
    {
        return ['mail', 'database'];
    }

    /**
     * Get the mail representation of the notification.
     */
    public function toMail(object $notifiable): MailMessage
    {
        $decisionColor = $this->decision === 'approved' ? '#10b981' : '#ef4444';
        $decisionText = ucfirst($this->decision);

        $message = (new MailMessage)
            ->subject('Fine Appeal Decision - ' . $decisionText)
            ->greeting('Hello ' . $notifiable->name . ',')
            ->line('Your fine appeal has been reviewed and a decision has been made.')
            ->line('**Appeal Decision:**')
            ->line('• **Decision:** ' . $decisionText)
            ->line('• **Reviewed By:** ' . $this->reviewedBy->name)
            ->line('• **Review Date:** ' . now()->format('M d, Y \a\t g:i A'))
            ->line('• **Decision Notes:** ' . $this->decisionNotes);

        if ($this->decision === 'approved') {
            $message->line('**Appeal Approved:**')
                ->line('Your appeal has been approved. The fine will be reversed and any deducted amounts will be refunded to your account.')
                ->line('• **Fine Amount:** $' . number_format($this->fine->fine_amount, 2) . ' ' . $this->fine->currency)
                ->line('• **Refund Amount:** $' . number_format($this->fine->amount_deducted, 2) . ' ' . $this->fine->currency)
                ->line('**Next Steps:**')
                ->line('1. The fine has been removed from your account')
                ->line('2. Any deducted amounts will be refunded within 3-5 business days')
                ->line('3. Your account status has been restored');
        } else {
            $message->line('**Appeal Denied:**')
                ->line('Your appeal has been denied. The fine remains in effect and will continue to be deducted from your earnings.')
                ->line('• **Fine Amount:** $' . number_format($this->fine->fine_amount, 2) . ' ' . $this->fine->currency)
                ->line('• **Remaining Balance:** $' . number_format($this->fine->remaining_amount, 2) . ' ' . $this->fine->currency)
                ->line('**Next Steps:**')
                ->line('1. The fine will continue to be deducted from future earnings')
                ->line('2. Contact support if you have additional questions')
                ->line('3. Focus on improving performance to avoid future infractions');
        }

        $message->action('View Fine Details', route('writer.fines.show', $this->fine->id))
            ->line('**Contact Support:**')
            ->line('If you have questions about this decision, please contact our support team.')
            ->salutation('Best regards, AcademicScribe Team');

        return $message;
    }

    /**
     * Get the array representation of the notification.
     *
     * @return array<string, mixed>
     */
    public function toArray(object $notifiable): array
    {
        return [
            'fine_id' => $this->fine->id,
            'decision' => $this->decision,
            'decision_notes' => $this->decisionNotes,
            'reviewed_by' => $this->reviewedBy->name,
            'reviewed_at' => now(),
            'fine_amount' => $this->fine->fine_amount,
            'remaining_amount' => $this->fine->remaining_amount,
            'amount_deducted' => $this->fine->amount_deducted,
            'type' => 'writer_fine_appeal_decision',
            'message' => 'Your fine appeal has been ' . $this->decision . '.',
        ];
    }

    /**
     * Get the notification's database type.
     */
    public function toDatabase(object $notifiable): array
    {
        return $this->toArray($notifiable);
    }
} 