<?php

namespace App\Notifications;

use App\Models\WriterInfraction;
use Illuminate\Bus\Queueable;
use Illuminate\Contracts\Queue\ShouldQueue;
use Illuminate\Notifications\Messages\MailMessage;
use Illuminate\Notifications\Notification;

class WriterInfractionEscalated extends Notification implements ShouldQueue
{
    use Queueable;

    protected $infraction;
    protected $escalationReason;

    /**
     * Create a new notification instance.
     */
    public function __construct(WriterInfraction $infraction, string $escalationReason)
    {
        $this->infraction = $infraction;
        $this->escalationReason = $escalationReason;
    }

    /**
     * Get the notification's delivery channels.
     *
     * @return array<int, string>
     */
    public function via(object $notifiable): array
    {
        return ['mail', 'database'];
    }

    /**
     * Get the mail representation of the notification.
     */
    public function toMail(object $notifiable): MailMessage
    {
        $severityColors = [
            'warning' => '#fbbf24',
            'minor' => '#f59e0b',
            'major' => '#dc2626',
            'critical' => '#7c2d12'
        ];

        $severityColor = $severityColors[$this->infraction->severity_level] ?? '#6b7280';

        return (new MailMessage)
            ->subject('⚠️ INFRACTION ESCALATED - Immediate Action Required')
            ->greeting('Hello ' . $notifiable->name . ',')
            ->line('**URGENT: Your infraction has been escalated due to the severity of the violation.**')
            ->line('**Escalation Details:**')
            ->line('• **Infraction ID:** #' . $this->infraction->id)
            ->line('• **Type:** ' . ucwords(str_replace('_', ' ', $this->infraction->infraction_type)))
            ->line('• **Current Severity:** ' . ucfirst($this->infraction->severity_level))
            ->line('• **Escalation Reason:** ' . $this->escalationReason)
            ->line('• **Escalated On:** ' . now()->format('M d, Y \a\t g:i A'))
            ->line('• **Original Infraction:** ' . $this->infraction->reported_at->format('M d, Y \a\t g:i A'))
            ->when($this->infraction->order_id, function ($message) {
                return $message->line('• **Related Order:** #' . $this->infraction->order_id);
            })
            ->line('**Immediate Consequences:**')
            ->line('1. Your account has been placed under review')
            ->line('2. New assignments may be restricted')
            ->line('3. Additional fines may be applied')
            ->line('4. Account suspension is possible')
            ->line('**Required Actions:**')
            ->line('1. **IMMEDIATE:** Submit a detailed response within 24 hours')
            ->line('2. **URGENT:** Contact support to discuss resolution options')
            ->line('3. **CRITICAL:** Review and address the underlying issues')
            ->action('Submit Response', route('writer.infractions.respond', $this->infraction->id))
            ->action('Contact Support', route('support.contact'))
            ->line('**Account Status:**')
            ->line('Your account is currently under disciplinary review. Failure to respond appropriately may result in account suspension.')
            ->line('**Appeal Rights:**')
            ->line('You have the right to appeal this escalation within 48 hours.')
            ->salutation('Best regards, AcademicScribe Disciplinary Team');
    }

    /**
     * Get the array representation of the notification.
     *
     * @return array<string, mixed>
     */
    public function toArray(object $notifiable): array
    {
        return [
            'infraction_id' => $this->infraction->id,
            'infraction_type' => $this->infraction->infraction_type,
            'severity_level' => $this->infraction->severity_level,
            'escalation_reason' => $this->escalationReason,
            'escalated_at' => now(),
            'reported_at' => $this->infraction->reported_at,
            'order_id' => $this->infraction->order_id,
            'type' => 'writer_infraction_escalated',
            'message' => 'Your infraction has been escalated - immediate action required.',
            'priority' => 'high',
        ];
    }

    /**
     * Get the notification's database type.
     */
    public function toDatabase(object $notifiable): array
    {
        return $this->toArray($notifiable);
    }
} 